/*!
 * WannaCry Analyser
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <https://www.gnu.org/licenses/>.
 * 
 * author: sizeof(cat) <sizeofcat AT riseup DOT net> https://sizeof.cat
 * url: https://sizeof.cat/project/wannacry-analyser/
 * copyright: 2016-2022
 * version 1.0.0
 * license: GPLv3 https://sizeof.cat/LICENSE.txt
 */
var wannacry_analyser = function() {
	// unused for now
	this.WC_PUBLIC_KEY = "00000000.pky";
	this.WC_PRIVATE_KEY = "00000000.dky";
	this.WC_ENCRYPTED_KEY = "00000000.eky";

	this.WC_SIG_LEN = 8;
	this.WC_ENCKEY_LEN = 256;
	this.WC_RSA_KEY_LEN = 2048;
	this.WC_AES_KEY_LEN = 16;
	this.WC_KEY_OFFSET = this.WC_SIG_LEN + 4;
	this.WC_DATA_OFFSET = this.WC_SIG_LEN + this.WC_ENCKEY_LEN + 4 + 4;
	this.WC_SIGNATURE = "WANACRY!";

	this.encrypted_key = null;
	this.encrypted_data = null;
	this.original_size = null;
	this.current_size = null;
	this.file_name = null;

	// unused for now
	this.decrypted_key = null;
	this.decrypted_data = null;

	this.ERR_INVALID_UNKNOWN = 1;
	this.ERR_INVALID_KEYSIZE = 2;
	this.ERR_INVALID_SIGNATURE = 3;
	this.ERR_INVALID_FILE = 4;
	this.ERR_INVALID_FILESIZE = 5;

	this.err_num = false;

	/*
	 * Initialize the library, nothing here for now.
	 */
	this.init = function() {
		// Nothing yet.
	};

	/*
	 * Reset the internal variables.
	 */
	this.reset = function() {
		this.encrypted_key = null;
		this.encrypted_data = null;
		this.original_size = null;
		this.current_size = null;
		this.file_name = null;
		this.err_num = false;
	};

	/*
	 * Parse the given file data if possible.
	 */
	this.parse_file = function(file_data, success_callback, error_callback) {
		var view;
		var signature;
		var unknown;
		var key;
		var keysize;
		var o1;
		var o2;
		if (file_data.byteLength <= 2147483648) {
			view = new DataView(file_data);
			signature = new Uint8Array(file_data, 0, this.WC_SIG_LEN);
			signature = new TextDecoder("utf-8").decode(signature);
			if (signature === this.WC_SIGNATURE) {
				keysize = view.getUint32(this.WC_SIG_LEN, true);
				if (keysize === this.WC_ENCKEY_LEN) {
					key = new Uint8Array(file_data, this.WC_KEY_OFFSET, this.WC_ENCKEY_LEN);
					if (key.length === this.WC_ENCKEY_LEN) {
						unknown = view.getUint32(this.WC_KEY_OFFSET + this.WC_ENCKEY_LEN, true);
						if (unknown === 3 || unknown === 4) {
							o1 = view.getUint32(this.WC_KEY_OFFSET + this.WC_ENCKEY_LEN + 4, true);
							o2 = view.getUint32(this.WC_KEY_OFFSET + this.WC_ENCKEY_LEN + 4 + 2, true);
							this.encrypted_key = key;
							this.encrypted_data = new Uint8Array(file_data, this.WC_DATA_OFFSET);
							this.original_size = (o1 << 32) + o2;
							this.current_size = file_data.byteLength;
							this.err_num = false;
							success_callback.call(this, this.get_data());
							return true;
						} else {
							this.err_num = this.ERR_INVALID_UNKNOWN;
						}
					} else {
						this.err_num = this.ERR_INVALID_KEYSIZE;
					}
				} else {
					this.err_num = this.ERR_INVALID_KEYSIZE;
				}
			} else {
				this.err_num = this.ERR_INVALID_SIGNATURE;
			}
		} else {
			this.err_num = this.ERR_INVALID_FILESIZE;
		}
		error_callback.call(this, this.error());
		return false;
	};

	/*
	 * Read the specified file via a FileReader instance.
	 */
	this.read_file = function(file, success_callback, error_callback) {
		var self = this;
		var file_reader = new FileReader();
		file_reader.onload = function(e) {
			self.reset();
			self.file_name = file.name;
			self.parse_file(e.target.result, success_callback, error_callback);
		};
		file_reader.onerror = function(e) {
			self.err_num = self.ERR_INVALID_FILE;
			error_callback.call(this, this.error());
		};
		file_reader.readAsArrayBuffer(file);
	};

	/*
	 * Show an error message.
	 */
	this.error = function() {
		switch (this.err_num) {
			case this.ERR_INVALID_UNKNOWN:
				return "Invalid unknown value, must be 3 or 4.";
				break;
			case this.ERR_INVALID_KEYSIZE:
				return "Invalid key size, must be " + this.WC_ENCKEY_LEN + '.';
				break;
			case this.ERR_INVALID_SIGNATURE:
				return "Invalid file signature, must be " + this.WC_SIGNATURE + '.';
				break;
			case this.ERR_INVALID_FILESIZE:
				return "Invalid file size, maximum can be 2MB."
				break;
			case this.ERR_INVALID_FILE:
				return "Invalid file.";
				break;
		}
	};

	this.encrypt = function() {
		// wishful thinking.
	};

	this.decrypt = function() {
		// wishful thinking.
	};

	/*
	 * Return the encrypted data payload.
	 */
	this.get_data = function() {
		return this.encrypted_data;
	};

	/*
	 * Return the initial filename (of the file that was uploaded).
	 */
	this.get_filename = function() {
		return this.file_name;
	};

	/*
	 * Return the encrypted AES key.
	 */
	this.get_key = function() {
		return this.encrypted_key;
	};

	/*
	 * Return the encrypted data payload formatted into a binary Blob.
	 */
	this.get_data_blob = function() {
		if (this.encrypted_data !== null) {
			return new Blob([this.encrypted_data], {
				type: 'application/octet-binary'
			});
		} else {
			return false;
		}
	};

	/*
	 * Return the parsed data from the specified file.
	 */
	this.get_data = function() {
		return {
			signature: this.WC_SIGNATURE,
			encrypted_data: this.encrypted_data,
			encrypted_key: this.encrypted_key,
			file_name: this.file_name,
			original_size: this.original_size,
			current_size: this.current_size
		};
	}

	/*
	 * Return the encrypted key formatted into a binary Blob.
	 */
	this.get_key_blob = function() {
		if (this.encrypted_key !== null) {
			return new Blob([this.encrypted_key], {
				type: 'application/octet-binary'
			});
		} else {
			return false;
		}
	};

	/*
	 * Boot and away we go.
	 */
	return this.init();
}
