/*!
 * Snek game.
 *
 * @author sizeof(cat) <sizeofcat AT riseup.net>
 * @copyright 2022 - 2023 sizeof(cat)
 * @version 1.2.0
 * @license GPLv3 https://sizeof.cat/LICENSE.txt
 * 
 *           o8o                                  .o88o.   .o                         .   o.   
 *           `"'                                  888 `"  .8'                       .o8   `8.  
 *  .oooo.o oooo    oooooooo  .ooooo.   .ooooo.  o888oo  .8'   .ooooo.   .oooo.   .o888oo  `8. 
 * d88(  "8 `888   d'""7d8P  d88' `88b d88' `88b  888    88   d88' `"Y8 `P  )88b    888     88 
 * `"Y88b.   888     .d8P'   888ooo888 888   888  888    88   888        .oP"888    888     88 
 * o.  )88b  888   .d8P'  .P 888    .o 888   888  888    `8.  888   .o8 d8(  888    888 .  .8' 
 * 8""888P' o888o d8888888P  `Y8bod8P' `Y8bod8P' o888o    `8. `Y8bod8P' `Y888""8o   "888" .8'  
 *                                                         `"                             "'   
 */

/*
 * Main game object.
 */
class game {
	/*
	 * Object constructor, gets ran on object initialization.
	 */
	 constructor (params) {

		/*
		 * Current scores.
		 */
		this.scores = {
			points: 0,
			moves: 0
		};

		/*
		 * Best scores.
		 */
		this.best_scores = {
			points: 0,
			moves: 0
		};

		/*
		 * Game configuration.
		 */
		this.config = {
			lines: 20,
			columns: 23,
			max_food: 4,
			username: ''
		};

		/*
		 * Game DOM elements.
		 */
		this.elements = {
			application: '.application',
			container: '.board',
			notification: '.notification',
			start: '.btn-start',
			stop: '.btn-stop',
			best_score: '.best_score',
			score: '.score',
			best_moves: '.best_moves',
			moves: '.moves',
			storage: 'snek'
		};

		/*
		 * Array containing the high scores list.
		 */
		this.highscores = [];

		/*
		 * Number of high scores to show in the list.
		 */
		this.MAX_HIGHSCORES = 3;

		/*
		 * Don't allow an username to be longer than this.
		 */
		this.MAX_USERNAME_LENGTH = 16;

		/*
		 * Game version.
		 */
		this.VERSION = '1.2.0';

		/*
		 * Various internal timeouts.
		 */
		this.restart_timeout = null;

		/*
		 * Private variables.
		 */
		this._snake = [[1, 2], [1, 1]];
		this._food = [];
		this._game_over = false;
		this._started = false;
		this._restart_mode = false;

		this.ref_el(this.elements.container).height(this.ref_el().height() - this.ref_el('footer').height() - this.ref_el('header').height() - 20);
		this.config.columns = Math.floor(this.ref_el(this.elements.container).width() / 26);
		this.config.lines = Math.floor(this.ref_el(this.elements.container).height() / 26);
		this.setup_from_storage();
		this.build_ui();
		this.attach_events();
		this.init();
	}

	get_random (max) {
		return Math.floor(Math.random() * max);
	}

	create_food () {
		let i = this.config.lines * this.config.columns;
		while (i > 0) {
			let row = 1 + this.get_random(this.config.lines);
			let col = 1 + this.get_random(this.config.columns);
			let is_ok = true;
			for (let pos of this._snake) {
				if (row == pos[0] && col == pos[1]) {
					is_ok = false;
				}
			}
			if (is_ok) {
				return [row, col];
			}
			i -= 1;
		}
	}

	draw_food () {
		for (let food of this._food) {
			if (food[0] && food[1]) {
				this.ref_el('#cell-' + food[0] + '-' + food[1]).addClass('food');
			}
		}
	}

	eat_food (new_head) {
		for (let food of this._food) {
			if (new_head[0] == food[0] && new_head[1] == food[1]) {
				this.ref_el('#cell-' + food[0] + '-' + food[1]).removeClass('food');
				this.scores.points += 1;
				const [row, col] = this.create_food();
				food[0] = row;
				food[1] = col;
				return true;
			}
		}
	}

	move (direction) {
		const curr = this._snake[0];
		const new_head = [curr[0], curr[1]];
		if (direction === 'up') {
			new_head[0] -= 1;
		} else if (direction === 'down') {
			new_head[0] += 1;
		} else if (direction === 'right') {
			new_head[1] += 1;
		} else if (direction === 'left') {
			new_head[1] -= 1;
		} else {
			return;
		}
		if (new_head[0] < 1) {
			return;
		}
		if (new_head[0] > this.config.lines) {
			return;
		}
		if (new_head[1] < 1) {
			return;
		}
		if (new_head[1] > this.config.columns) {
			return;
		}
		for (let pos of this._snake) {
			if (new_head[0] == pos[0] && new_head[1] == pos[1]) {
				this.game_over();
				this._game_over = true;
				return;
			}
		}
		this.scores.moves += 1;
		this._snake.unshift(new_head);
		if (!this.eat_food(new_head)) {
			this._snake.pop();
		}
	}

	draw () {
		let head = true;
		for (let pos of this._snake) {
			if (head) {
				this.ref_el('#cell-' + pos[0] + '-' + pos[1]).removeClass('food snakebody').addClass('snakehead');
				head = false;
			} else {
				this.ref_el('#cell-' + pos[0] + '-' + pos[1]).removeClass('snakehead').addClass('snakebody');
			}
		}
	}

	build_ui () {
		let _t = '';
		for (let row = 1; row <= this.config.lines; row++) {
			for (let col = 1; col <= this.config.columns; col++) {
				_t += '<div class="cell" id="cell-' + row + '-' + col + '"></div>';
			}
		}
		this.ref_el(this.elements.container).html(_t);
		for (let i = 1; i <= this.config.max_food; i++) {
			this._food.push(this.create_food());
		}
		this.draw_food();
		this.draw();
		this.ref_el('.game-username').attr('maxlength', this.MAX_USERNAME_LENGTH);
		this.ref_el('.game-version').html('v' + this.VERSION);
	}

	attach_events () {
		let self = this;

		document.body.onkeydown = function(ev) {
			if (!self._started || self._game_over) {
				return;
			}
			if (ev.key.toLowerCase() === 'w') {
				self.move('up');
			} else if (ev.key.toLowerCase() === 's') {
				self.move('down');
			} else if (ev.key.toLowerCase() === 'd') {
				self.move('right');
			} else if (ev.key.toLowerCase() === 'a') {
				self.move('left');
			} else {
				return;
			}
			$('.board .cell').removeClass('food').removeClass('snakebody');
			self.draw_food();
			self.draw();
			self.ref_el(self.elements.score).html(self.scores.points);
			self.ref_el(self.elements.moves).html(self.scores.moves);
		};
		this.ref_el().on('click', this.elements.start, function () {
			self.restart();
			self._started = true;
			return false;
		}).on('click', '.btn-restart', function () {
			$('.application .panel').hide();
			$('.application .content').fadeIn();
			return false;
		}).on('click', this.elements.stop, function() {
			if (self._restart_mode === false) {
				$(this).addClass('attention').html('Are you sure?');
				self._restart_mode = true;
				self.restart_timeout = setTimeout(function () {
					self.ref_el(self.elements.stop).removeClass('attention').html('New Game');
					clearTimeout(self.restart_timeout);
					self._restart_mode = false;
				}, 5000);
			} else {
				self.stop();
			}
			return false;
		}).on('click', '.btn-settings', function () {
			$('.application .panel').hide();
			$('.application .panel.settings').fadeToggle();
		}).on('click', '.btn-scores', function () {
			self.ref_el('.scoreboard').empty();
			self.ref_el('.panel').hide();
			self.ref_el('.panel.scores').fadeToggle();
			if (self.highscores.length > 0) {
				self.rank_high_scores();
				for (let i = 0; i < self.highscores.length; i++) {
					self.ref_el('.scoreboard').append('<li><span class="high-score-score">' + self.highscores[i].score + '</span>  <span class="high-score-name">' + self.highscores[i].name + '</span></li>');
					self.ref_el('.loading').hide();
					self.ref_el('.scoreboard').show();
				}
			} else {
				self.ref_el('.loading').html('... no high scores yet ...');
			}
			return false;
		}).on('click', '.btn-about', function () {
			$('.application .panel').hide();
			$('.application .panel.about').fadeToggle();
		}).on('keyup', '.game-username', function() {
			let username = $('.game-username').val();
			self.change_username(username);
		});
	}

	/*
	 * Sort the highscores and keep just the top X.
	 */
	rank_high_scores () {
		this.highscores.sort(function(a, b) {
			let keyA = new Date(a.score);
			let keyB = new Date(b.score);
			if (keyA > keyB) {
				return -1;
			}
			if (keyA < keyB) {
				return 1;
			}
			return 0;
		});
		if (this.highscores.length >= this.MAX_HIGHSCORES) {
			this.highscores.splice(this.MAX_HIGHSCORES, this.highscores.length - this.MAX_HIGHSCORES);
		}
	}

	/*
	 * Save the game data.
	 */
	save () {
		if (this.scores.points > this.best_scores.points) {
			this.best_scores.points = this.scores.points;
			this.set_storage('bestscore', this.scores.points);
			this.ref_el(this.elements.best_score).html(this.scores.points);
		}
		if (this.scores.moves > this.best_scores.moves) {
			this.best_scores.moves = this.scores.moves;
			this.set_storage('bestmoves', this.scores.moves);
			this.ref_el(this.elements.best_moves).html(this.scores.moves);
		}
		if (this.scores.points > 0) {
			this.highscores.push({
				name: this.config.username,
				score: this.scores.points
			});
		}
		this.set_storage('highscores', JSON.stringify(this.highscores));
	}

	/*
	 * Update the user name.
	 */
	change_username (username) {
		if (username.length > this.MAX_USERNAME_LENGTH) {
			username = username.substring(0, this.MAX_USERNAME_LENGTH);
		}
		this.config.username = username;
		this.set_storage('name', username);
	}

	/*
	 * Load initial data from browser localStorage.
	 */
	setup_from_storage () {
		let local_best_score = this.get_storage('bestscore');
		if (local_best_score != null) {
			this.best_scores.points = local_best_score;
		}
		this.ref_el(this.elements.best_score).html(this.best_scores.points);
		let local_best_moves = this.get_storage('bestmoves');
		if (local_best_moves != null) {
			this.best_scores.moves = local_best_moves;
		}
		this.ref_el(this.elements.best_moves).html(this.best_scores.moves);
		if (this.get_storage('name') == null) {
			this.ref_el('.panel.settings').show();
		} else {
			this.config.username = this.get_storage('name');
		}
		this.ref_el('.game-username').val(this.config.username);
		let local_high_scores = this.get_storage('highscores');
		if (local_high_scores != null) {
			this.highscores = JSON.parse(local_high_scores);
		}
	}

	/*
	 * Save a key with the specified value to the storage.
	 */
	set_storage (key, value) {
		localStorage[this.elements.storage + '.' + key] = value;
	}

	/*
	 * Get a key from the storage.
	 */
	get_storage (key) {
		return localStorage[this.elements.storage + '.' + key];
	}

	/*
	 * Return a DOM element referenced by the game main DOM element.
	 */
	ref_el (element) {
		if (typeof element === 'undefined') {
			return $(this.elements.application);
		} else {
			return $(this.elements.application + ' ' + element);
		}
	}

	cleanup () {
		this.scores.moves = 0;
		this.scores.points = 0;
	}

	init () {
		this.cleanup();
		this.ref_el(this.elements.score).html(this.scores.points);
		this.ref_el(this.elements.moves).html(this.scores.moves);
		if (this.config.username === '') {
			this.config.username = 'user-' + Math.floor(Math.random() * 9999);
			this.set_storage('name', this.config.username);
			this.ref_el('.game-username').val(this.config.username);
		}
	}

	game_over () {
		this.notify('GAME OVER<br />Score: ' + this.scores.points + '<br />Max moves: ' + this.scores.moves, true);
		this.save();
		this.ref_el('footer').hide();
	}

	/*
	 * Reset the game UI to initial state.
	 */
	reset_ui () {
		this.ref_el('.content').hide();
		this.ref_el(this.elements.notification).html('');
		this.ref_el(this.elements.score).html('0');
		this.ref_el(this.elements.moves).html('0');
		this.ref_el('footer').hide();
	};

	/*
	 * Restart the game.
	 */
	restart () {
		this.ref_el('footer').show();
		$('.application .content').hide();
		this.init();
	}

	/*
	 * Stop the game.
	 */
	stop () {
		this.ref_el(this.elements.stop).removeClass('attention').html('New Game');
		this.reset_ui();
		this.ref_el('.panel').hide();
		this.ref_el('.content').fadeIn();
		this._restart_mode = false;
		this._started = false;
		clearTimeout(this.restart_timeout);
	}

	/*
	 * Perform a game notification.
	 */
	notify (message) {
		this.ref_el(this.elements.notification).html('<div>' + message + '</div>');
	}
}

$(document).ready(function() {
	new game();
});
