/*!
 * Sapper game.
 *
 * @author sizeof(cat) <sizeofcat AT riseup.net>
 * @copyright 2016 - 2023 sizeof(cat)
 * @version 1.4.0
 * @license GPLv3 https://sizeof.cat/LICENSE.txt
 * 
 *           o8o                                  .o88o.   .o                         .   o.   
 *           `"'                                  888 `"  .8'                       .o8   `8.  
 *  .oooo.o oooo    oooooooo  .ooooo.   .ooooo.  o888oo  .8'   .ooooo.   .oooo.   .o888oo  `8. 
 * d88(  "8 `888   d'""7d8P  d88' `88b d88' `88b  888    88   d88' `"Y8 `P  )88b    888     88 
 * `"Y88b.   888     .d8P'   888ooo888 888   888  888    88   888        .oP"888    888     88 
 * o.  )88b  888   .d8P'  .P 888    .o 888   888  888    `8.  888   .o8 d8(  888    888 .  .8' 
 * 8""888P' o888o d8888888P  `Y8bod8P' `Y8bod8P' o888o    `8. `Y8bod8P' `Y888""8o   "888" .8'  
 *                                                         `"                             "'   
 */

/*
 * Main game object.
 */
class game {

	/*
	 * Current scores.
	 */
	scores = {
		points: 0
	}

	/*
	 * Best scores.
	 */
	best_scores = {
		points: 0
	}

	/*
	 * Array containing the high scores list.
	 */
	highscores = [];

	/*
	 * Number of high scores to show in the list.
	 */
	MAX_HIGHSCORES = 3;

	/*
	 * Don't allow an username to be longer than this.
	 */
	MAX_USERNAME_LENGTH = 16;

	/*
	 * Game version.
	 */
	VERSION = '1.4.0';

	/*
	 * Game DOM elements.
	 */
	elements = {
		application: '.application',
		container: '.game-container',
		notification: '.notification',
		start: '.btn-start',
		stop: '.btn-stop',
		best_score: '.best_score',
		score: '.score',
		flags: '.flags',
		mines: '.mines',
		time: '.time',
		storage: 'sapper'
	}

	/*
	 * Game configuration.
	 */
	config = {
		username: '',
		difficulty: 9,
		cell_size: 0,
		cells: 14,
		width: 0,
		cell_bg_color: '#444',
		cell_hover_color: '#666',
		cell_fg_color: '#222',
		cell_radius: 10
	}

	/*
	 * Various internal timeouts.
	 */
	restart_timeout = null;
	clock = null;
	restart = null;

	/*
	 * Private variables.
	 */
	_restart_mode = false;
	started = false;
	gameover = false;
	canvas = null;
	ctx = null;
	num_mines = 0;
	num_flags = 0;
	time = 0;
	map_mines = '';
	map_flags = '';
	map_revealed = '';
	previous = new Array(2);
	img_mine = './img/bomb.png';
	img_flag = './img/flag.png';

	/*
	 * Object constructor, gets ran on object initialization.
	 */
	 constructor (params) {
		this.setup_from_storage();
		this.build_ui();
		this.init();
	}

	build_ui () {
		this.ref_el('.game-username').attr('maxlength', this.MAX_USERNAME_LENGTH);
		this.ref_el('.game-version').html('v' + this.VERSION);
	}

	/*
	 * Return a DOM element referenced by the game main DOM element.
	 */
	ref_el (element) {
		if (typeof element === 'undefined') {
			return $(this.elements.application);
		} else {
			return $(this.elements.application + ' ' + element);
		}
	}

	init () {
		let self = this;
		this.canvas = $('#board');
		let _w = this.ref_el().height() - this.ref_el('footer').height() - this.ref_el('header').height() - 20;
		let _h = this.ref_el().width();
		if (_w >= _h) {
			this.config.width = _h;
		} else {
			this.config.width = _w;
		}
		this.canvas[0].width = this.config.width;
		this.canvas[0].height = this.config.width;
		this.ctx = this.canvas[0].getContext("2d");
		this.ctx.clearRect(0, 0, this.config.width, this.config.width);
		this.config.cell_size = Math.floor(this.config.width / this.config.cells);
		this.config.cell_radius = this.config.cell_size / 4;
		this.map_mines = new Array(this.config.cells);
		this.map_flags = new Array(this.config.cells);
		this.map_revealed = new Array(this.config.cells);
		this.ctx.font = this.config.cell_size / 2 + 'px Eczar';
		this.ref_el(this.elements.container).height(this.ref_el().height() - this.ref_el('footer').height() - this.ref_el('header').height() - 20);
		this.canvas.on({
			mouseup: function(event) {
				self.click(event);
			},
			mousemove: function(event) {
				self.hover(event);
			},
			contextmenu: function(event) {
				return false;
			}
		});
		let images = new Array();
		images[0] = new Image();
		images[0].src = this.img_mine;
		images[1] = new Image();
		images[1].src = this.img_flag;
		this.setup();
		this.ref_el().on('click', self.elements.start, function() {
			$('.application .content').hide();
			self.reset();
			return false;
		}).on('click', '.btn-scores', function() {
			self.ref_el('.scoreboard').empty();
			self.ref_el('.panel').hide();
			self.ref_el('.panel.scores').fadeToggle();
			if (self.highscores.length > 0) {
				self.rank_high_scores();
				for (let i = 0; i < self.highscores.length; i++) {
					self.ref_el('.scoreboard').append('<li><span class="high-score-score">' + self.highscores[i].score + '</span>  <span class="high-score-name">' + self.highscores[i].name + '</span></li>');
					self.ref_el('.loading').hide();
					self.ref_el('.scoreboard').show();
				}
			} else {
				self.ref_el('.loading').html('... no high scores yet ...');
			}
			return false;
		}).on('change', '.game-difficulty', function() {
			self.config.difficulty = $(this).val();
		}).on('keyup', '.game-username', function() {
			let username = $('.game-username').val();
			self.change_username(username);
		}).on('click', '.btn-settings', function() {
			self.ref_el('.panel').hide();
			self.ref_el('.panel.settings').fadeToggle();
			return false;
		}).on('click', '.btn-about', function() {
			self.ref_el('.panel').hide();
			self.ref_el('.panel.about').fadeToggle();
			return false;
		}).on('click', self.elements.stop, function() {
			if (self._restart_mode === false) {
				$(this).addClass('attention').html('Are you sure?');
				self._restart_mode = true;
				self.restart_timeout = setTimeout(function () {
					self.ref_el(self.elements.stop).removeClass('attention').html('New Game');
					clearTimeout(self.restart_timeout);
					self._restart_mode = false;
				}, 5000);
			} else {
				self.stop();
			}
			return false;
		});
	}

	/*
	 * Stop the game.
	 */
	stop = function() {
		this.ref_el(this.elements.stop).removeClass('attention').html('New Game');
		this.reset_ui();
		this.ref_el('.panel').hide();
		this.ref_el('.content').fadeIn();
		this._restart_mode = false;
		clearTimeout(this.restart_timeout);
	}

	/*
	 * Reset the game UI to initial state.
	 */
	reset_ui = function() {
		this.ref_el('.content').hide();
		this.ref_el(this.elements.notification).html('');
		this.ref_el(this.elements.score).html('0');
		this.ref_el(this.elements.flags).html('0');
		this.ref_el(this.elements.mines).html('0');
		this.ref_el(this.elements.time).html('1');
		this.ref_el('footer').hide();
	}

	/*
	 * Update the user name.
	 */
	change_username = function(username) {
		if (username.length > self.MAX_USERNAME_LENGTH) {
			username = username.substring(0, self.MAX_USERNAME_LENGTH);
		}
		this.config.username = username;
		this.set_storage('name', username);
	}

	reset () {
		if (this.config.username === '') {
			this.config.username = 'user-' + Math.floor(Math.random() * 9999);
			this.set_storage('name', this.config.username);
			this.ref_el('.game-username').val(this.config.username);
		}
		this.ref_el('.notification').html('');
		window.clearInterval(this.clock);
		window.clearInterval(this.restart);
		this.ctx.clearRect(0, 0, this.config.width, this.config.width);
		this.gameover = false;
		this.started = false;
		this.num_mines = 0;
		this.num_flags = 0;
		this.time = 0;
		this.map_mines = new Array(this.config.cells);
		this.map_flags = new Array(this.config.cells);
		this.map_revealed = new Array(this.config.cells);
		this.ref_el('.time').html('1');
		this.ref_el('.flags').html('0');
		this.ref_el('.mines').html('0');
		this.setup();
		this.ref_el('footer').show();
	}
	
	setup () {
		for (let k = 0; k < this.config.cells; k++) {
			this.map_flags[k] = Array(this.config.cells);
			this.map_revealed[k] = Array(this.config.cells);
		}
		this.ctx.strokeStyle = this.config.cell_fg_color;
		this.ctx.fillStyle = this.config.cell_bg_color;
		this.board();
	}

	timer () {
		let self = this;
		this.clock = setInterval(function() {
			self.time++;
			self.ref_el('.time').html(self.time);
		}, 1000);
	}

	board () {
		for (let i = 0; i < this.config.cells; i++) {
			for (let j = 0; j < this.config.cells; j++) {
				this.draw_mine(i, j);
			}
		}
	}

	draw_mine (xx, yy) {
		let width = this.config.cell_size - 1;
		let x = xx * this.config.cell_size;
		let y = yy * this.config.cell_size;
		this.ctx.beginPath();
		this.ctx.moveTo(x + this.config.cell_radius, y);
		this.ctx.lineTo(x + width - this.config.cell_radius, y);
		this.ctx.quadraticCurveTo(x + width, y, x + width, y + this.config.cell_radius);
		this.ctx.lineTo(x + width, y + width - this.config.cell_radius);
		this.ctx.quadraticCurveTo(x + width, y + width, x + width - this.config.cell_radius, y + width);
		this.ctx.lineTo(x + this.config.cell_radius, y + width);
		this.ctx.quadraticCurveTo(x, y + width, x, y + width - this.config.cell_radius);
		this.ctx.lineTo(x, y + this.config.cell_radius);
		this.ctx.quadraticCurveTo(x, y, x + this.config.cell_radius, y);
		this.ctx.closePath();
		this.ctx.stroke();
		this.ctx.fill();
	}

	is (what, x, y) {
		let p = {
			revealed: this.map_revealed,
			mine: this.map_mines,
			flag: this.map_flags
		};
		if (typeof p[what][x] !== 'undefined' && typeof p[what][x][y] !== 'undefined' && p[what][x][y] > -1) {
			return true;
		} else {
			return false;
		}
	}
	
	click (e) {
		let self = this;
		if (self.gameover) {
			return false;
		}
		let x = Math.floor((e.pageX - self.canvas.offset().left) / self.config.cell_size);
		let y = Math.floor((e.pageY - self.canvas.offset().top) / self.config.cell_size);
		if (x < self.config.cells && y < self.config.cells) {
			let l = self.map_revealed[x][y] ? 1 : -1;
			if (e.which === 1 && self.map_flags[x][y] !== 1 && self.config.difficulty !== 0) {
				if (self.started === false) {
					self.board();
					self.timer();
					do {
						self.generate_mines(self.map_mines);
					} while (self.map_mines[x][y] === -1);
					self.ref_el('.mines').html(self.num_mines);
					self.started = true;
				}
				self.index(x, y);
			} else if (e.which === 3 && self.is('revealed', x, y)) {
				let num = 0;
				let neighbours = new Array();
				let xArr = [x, x + 1, x - 1];
				let yArr = [y, y + 1, y - 1];
				for (let a = 0; a < 3; a++) {
					for (let b = 0; b < 3; b++) {
						if (self.is('flag', xArr[a], yArr[b])) {
							num++;
						} else {
							neighbours.push([xArr[a], yArr[b]]);
						}
					}
				}
				if (num === self.map_mines[x][y]) {
					$.each(neighbours, function() {
						self.index(this[0], this[1]);
					});
				}
			} else if (e.which === 3 && l < 0 && self.started !== false) {
				let flag = new Image();
				flag.src = self.img_flag;
				flag.onload = function() {
					self.flag(flag, x, y);
				};
			}
		}
	}

	hover (e) {
		let self = this;
		if (!self.gameover) {
			let x = Math.floor((e.pageX - self.canvas.offset().left) / self.config.cell_size);
			let y = Math.floor((e.pageY - self.canvas.offset().top) / self.config.cell_size);
			if (x < self.config.cells && y < self.config.cells) {
				let l = self.map_revealed[x][y] ? 1 : -1;
				let f = self.map_flags[x][y] ? 1 : -1;
				let pX = self.previous[0];
				let pY = self.previous[1];
				if (typeof pX !== 'undefined' && self.map_revealed[pX][pY] !== 1 && self.map_flags[pX][pY] !== 1) {
					self.ctx.fillStyle = self.config.cell_bg_color;
					self.draw_mine(self.previous[0], self.previous[1]);
				}
				if (l < 0 && f < 0 && self.started) {
					self.ctx.fillStyle = self.config.cell_hover_color;
					self.draw_mine(x, y);
					self.previous[0] = x;
					self.previous[1] = y;
				}
			}
		}
	}

	index (x, y) {
		let self = this;
		if (x >= 0 && y >= 0 && x < self.config.cells && y < self.config.cells && self.map_mines[x] !== undefined) {
			let l = (self.map_revealed[x][y]) ? 1 : -1;
			if (!self.is('revealed', x, y)) {
				self.map_revealed[x][y] = 1;
				if (self.map_mines[x][y] !== -1) {
					let alpha = 0.1;
					let square_fade = setInterval(function() {
						self.ctx.strokeStyle = self.config.cell_fg_color;
						self.ctx.fillStyle = 'rgba(255, 255, 255,' + alpha + ')';
						self.draw_mine(x, y);
						if (self.map_mines[x][y] !== -1) {
							let color_map = ['none', 'blue', 'green', 'red', 'black', 'orange', 'cyan'];
							if (color_map[self.map_mines[x][y]] !== 'none') {
								self.ctx.fillStyle = color_map[self.map_mines[x][y]];
								self.ctx.fillText(self.map_mines[x][y], (x * self.config.cell_size) + (self.config.cell_size / 3), (y * self.config.cell_size) + (self.config.cell_size / 1.7));
							}
						}
						alpha = alpha + .1;
						if (alpha > 1) {
							window.clearInterval(square_fade);
						}
					}, 50);
				} else {
					let mine = new Image();
					mine.src = self.img_mine;
					mine.onload = function() {
						self.show_mines(mine);
					};
				}
				if (self.map_mines[x][y] === 0) {
					for (let i = -1; i <= 1; i++) {
						for (let j = -1; j <= 1; j++) {
							if (l < 0 && x + i >= 0 && y + j >= 0 && x + i <= self.config.cells && y + j <= self.config.cells) {
								self.index(x + i, y + j);
							}
						}
					}
				}
			}
		}
	}

	flag (flag, x, y) {
		if (this.map_flags[x][y] !== 1) {
			this.ctx.drawImage(flag, x * this.config.cell_size, y * this.config.cell_size, this.config.cell_size, this.config.cell_size);
			this.map_flags[x][y] = 1;
			this.num_flags++;
		} else {
			let img = this.ctx.createImageData(this.config.cell_size, this.config.cell_size);
			for (let i = img.data.length; --i >= 0; ) {
				img.data[i] = 0;
			}
			this.ctx.putImageData(img, x * this.config.cell_size, y * this.config.cell_size);
			this.ctx.strokeStyle = this.config.cell_fg_color;
			this.ctx.fillStyle = this.config.cell_bg_color;
			this.draw_mine(x, y);
			this.map_flags[x][y] = 0;
			this.num_flags--;
		}
		this.ref_el('.mines').html((this.num_mines - this.num_flags));
		this.ref_el('.flags').html(this.num_flags);
		this.won();
	}

	won () {
		let count = 0;
		for (let i = 0; i < this.config.cells; i++) {
			for (let j = 0; j < this.config.cells; j++) {
				if ((this.map_flags[i][j] === 1) && (this.map_mines[i][j] === -1)) {
					count++;
				}
			}
		}
		if (count === this.num_mines) {
			this.gameover = true;
			let score = Math.floor(this.num_mines * this.time / this.config.difficulty);
			if (score > this.best_scores.points) {
				this.best_scores.points = score;
				this.save();
			}
			self.notify('GAME OVER<br />Score: ' + score);
			self.save();
			window.clearInterval(this.clock);
		}
	}

	/*
	 * Perform a game notification.
	 */
	notify = function(message) {
		this.ref_el(this.elements.notification).html('<div>' + message + '</div>');
	}

	/*
	 * Sort the highscores and keep just the top X.
	 */
	rank_high_scores () {
		this.highscores.sort(function(a, b) {
			let keyA = new Date(a.score);
			let keyB = new Date(b.score);
			if (keyA > keyB) {
				return -1;
			}
			if (keyA < keyB) {
				return 1;
			}
			return 0;
		});
		if (this.highscores.length >= this.MAX_HIGHSCORES) {
			this.highscores.splice(this.MAX_HIGHSCORES, this.highscores.length - this.MAX_HIGHSCORES);
		}
	}

	/*
	 * Save the game data.
	 */
	save () {
		if (this.scores.points > this.best_scores.points) {
			this.best_scores.points = this.scores.points;
			this.set_storage('bestscore', this.scores.points);
			this.ref_el(this.elements.best_score).html(this.scores.points);
		}
		if (this.scores.points > 0) {
			this.highscores.push({
				name: this.config.username,
				score: this.scores.points
			});
		}
		this.set_storage('highscores', JSON.stringify(this.highscores));
	}

	generate_mines () {
		for (let i = 0; i < this.config.cells; i++) {
			this.map_mines[i] = new Array(this.config.cells);
			for (let j = 0; j < this.config.cells; j++) {
				this.map_mines[i][j] = Math.floor((Math.random() * this.config.difficulty) - 1);
				if (this.map_mines[i][j] > 0) {
					this.map_mines[i][j] = 0;
				}
			}
		}
		this.calc_mines();
	}

	calc_mines () {
		this.num_mines = 0;
		for (let i = 0; i < this.config.cells; i++) {
			for (let j = 0; j < this.config.cells; j++) {
				if (this.map_mines[i][j] === -1) {
					let xArr = [i, i + 1, i - 1];
					let yArr = [j, j + 1, j - 1];
					for (let a = 0; a < 3; a++) {
						for (let b = 0; b < 3; b++) {
							if (this.is('mine', xArr[a], yArr[b])) {
								this.map_mines[xArr[a]][yArr[b]]++;
							}
						}
					}
					this.num_mines++;
				}
			}
		}
	}

	show_mines (mine) {
		for (let i = 0; i < this.config.cells; i++) {
			for (let j = 0; j < this.config.cells; j++) {
				if (this.map_mines[i][j] === -1) {
					this.ctx.drawImage(mine, i * this.config.cell_size, j * this.config.cell_size, this.config.cell_size, this.config.cell_size);
				}
			}
		}
		this.gameover = true;
		this.notify('GAME OVER');
		window.clearInterval(this.clock);
	}

	/*
	 * Load initial data from browser localStorage.
	 */
	setup_from_storage () {
		let local_best_score = this.get_storage('bestscore');
		if (local_best_score != null) {
			this.best_scores.points = local_best_score;
		}
		this.ref_el(this.elements.best_score).html(this.best_scores.points);
		if (this.get_storage('name') == null) {
			this.ref_el('.panel.settings').show();
		} else {
			this.config.username = this.get_storage('name');
		}
		this.ref_el('.game-username').val(this.config.username);
		let local_high_scores = this.get_storage('highscores');
		if (local_high_scores != null) {
			this.highscores = JSON.parse(local_high_scores);
		}
	}

	/*
	 * Save a key with the specified value to the storage.
	 */
	set_storage (key, value) {
		localStorage[this.elements.storage + '.' + key] = value;
	}

	/*
	 * Get a key from the storage.
	 */
	get_storage (key) {
		return localStorage[this.elements.storage + '.' + key];
	}
}

$(document).ready(function() {
	new game();
});
