/*!
 * Pome game.
 *
 * @author sizeof(cat) <sizeofcat AT riseup.net>
 * @copyright 2019 - 2023 sizeof(cat)
 * @version 3.2.0
 * @license GPLv3 https://sizeof.cat/LICENSE.txt
 * 
 *           o8o                                  .o88o.   .o                         .   o.   
 *           `"'                                  888 `"  .8'                       .o8   `8.  
 *  .oooo.o oooo    oooooooo  .ooooo.   .ooooo.  o888oo  .8'   .ooooo.   .oooo.   .o888oo  `8. 
 * d88(  "8 `888   d'""7d8P  d88' `88b d88' `88b  888    88   d88' `"Y8 `P  )88b    888     88 
 * `"Y88b.   888     .d8P'   888ooo888 888   888  888    88   888        .oP"888    888     88 
 * o.  )88b  888   .d8P'  .P 888    .o 888   888  888    `8.  888   .o8 d8(  888    888 .  .8' 
 * 8""888P' o888o d8888888P  `Y8bod8P' `Y8bod8P' o888o    `8. `Y8bod8P' `Y888""8o   "888" .8'  
 *                                                         `"                             "'   
 */

/*
 * Main game object.
 */
class game {

	/*
	 * Current scores.
	 */
	scores = {
		points: 0,
		combo: 0
	}

	/*
	 * Best scores.
	 */
	best_scores = {
		points: 0,
		combo: 0
	}

	/*
	 * Game configuration.
	 */
	config = {
		lines: 0,
		columns: 0,
		size: 48,
		theme: 0,
		blocks: 5,
		multiplier: 0,
		username: ''
	}

	/*
	 * Board sizes for desktop and mobile.
	 */
	boards = {
		desktop: {
			lines: 9,
			columns: 10
		},
		mobile: {
			lines: 12,
			columns: 8
		}
	}

	/*
	 * Game DOM elements.
	 */
	elements = {
		application: '.application',
		container: '#board',
		notification: '.notification',
		start: '.btn-start',
		stop: '.btn-stop',
		best_score: '.best_score',
		score: '.score',
		best_combo: '.best_combo',
		combo: '.combo',
		moves: '.moves',
		storage: 'pome'
	}

	/*
	 * Array containing the high scores list.
	 */
	highscores = [];

	/*
	 * List of game themes.
	 */
	themes = [
		{
			name: 'spring'
		}, {
			name: 'food'
		}, {
			name: 'animals'
		}, {
			name: 'fruity'
		}, {
			name: 'camping'
		}, {
			name: 'numbers'
		}, {
			name: 'pastel'
		}, {
			name: 'letters'
		}, {
			name: 'game of thrones'
		}
	]

	/*
	 * Number of high scores to show in the list.
	 */
	MAX_HIGHSCORES = 3;

	/*
	 * Score gets multiplied by the value here.
	 */
	SCORE_MULTIPLIER = 1;

	/*
	 * Don't allow an username to be longer than this.
	 */
	MAX_USERNAME_LENGTH = 16;

	/*
	 * Game version.
	 */
	VERSION = '3.2.0';

	/*
	 * Show a hint after the specified time has passed (in ms).
	 */
	TIMEOUT_HINT = 10000;

	/*
	 * Time in which successful moves count as light-speed moves and combos
	 * get added (in ms).
	 */
	TIMEOUT_LIGHTSPEED = 1500;

	/*
	 * Various internal timeouts.
	 */
	hint_timeout = null;
	lightspeed_timeout = null;
	restart_timeout = null;

	/*
	 * Private variables.
	 */
	_hint_mode = false;
	_restart_mode = false;
	_block_array = [];
	_block_array_up = [];
	_init_x = 0;
	_init_y = 0;
	_test_horizontal = [];
	_test_vertical = [];
	_test_tab = [];
	_start_move = false;
	_forbid_move = false;
	_block_line = 0;
	_block_column = 0;
	_selected_element = null;
	_block_element = null;

	/*
	 * Object constructor, gets ran on object initialization.
	 */
	constructor() {
		this.resize();
		this.setup_from_storage();
		this.build_ui();
		this.attach_events();
	}

	build_ui = function() {
		this.ref_el('.game-username').attr('maxlength', this.MAX_USERNAME_LENGTH);
		for (let i = 0; i < this.themes.length; i++) {
			let id = i;
			if (this.themes[i].id !== undefined) {
				id = this.themes[i].id;
			}
			this.ref_el('.game-theme').append('<option value="' + id + '">' + this.themes[i].name + '</option>');
		}
		this.ref_el('.game-version').html('v' + this.VERSION);
	}

	attach_events = function() {
		let self = this;
		$(window).on('resize', function() {
			self.resize(true);
		});
		this.ref_el().on('dragstart', '.block', function(event) {
			event.preventDefault();
		}).on('touchmove mousemove', self.elements.notification, function(event) {
			event.preventDefault();
		}).on('touchstart mousedown', '.block', function(event) {
			if (!self._start_move && !self._forbid_move) {
				self.drag_move = false;
				self._block_element = $(this);
				self._block_element.css('z-index', 20);
				self._block_line = parseInt(self._block_element.attr('data-line'));
				self._block_column = parseInt(self._block_element.attr('data-col'));
				if (event.originalEvent.type === 'touchstart') {
					self._init_x = event.originalEvent.touches[0].clientX;
					self._init_y = event.originalEvent.touches[0].clientY;
				}
				if (event.originalEvent.type === 'mousedown') {
					self._init_x = event.originalEvent.clientX;
					self._init_y = event.originalEvent.clientY;
				}
				self._start_move = true;
			}
		}).on('touchmove mousemove', self.elements.container, function(event) {
			event.preventDefault();
			if (self._start_move) {
				let distance_x;
				let distance_y;
				if (event.originalEvent.type === 'touchmove') {
					distance_x = event.originalEvent.touches[0].clientX - self._init_x;
					distance_y = event.originalEvent.touches[0].clientY - self._init_y;
				}
				if (event.originalEvent.type === 'mousemove') {
					distance_x = event.originalEvent.clientX - self._init_x;
					distance_y = event.originalEvent.clientY - self._init_y;
				}
				if (Math.abs(distance_x) > Math.abs(distance_y)) {
					if (distance_x > self.config.size / 2) {
						if (self._block_column < self.config.columns - 1) {
							self.drag_move = true;
							self.ref_el('.block').removeClass('click near');
							self.do_move(self._block_line, self._block_column, self._block_line, self._block_column + 1);
						}
					}
					if (distance_x < -self.config.size / 2) {
						if (self._block_column > 0) {
							self.drag_move = true;
							$('.block').removeClass('click near');
							self.do_move(self._block_line, self._block_column, self._block_line, self._block_column - 1);
						}
					}
				} else {
					if (distance_y > self.config.size / 2) {
						if (self._block_line < self.config.lines - 1) {
							self.drag_move = true;
							self.ref_el('.block').removeClass('click near');
							self.do_move(self._block_line, self._block_column, self._block_line + 1, self._block_column);
						}
					}
					if (distance_y < -self.config.size / 2) {
						if (self._block_line > 0) {
							self.drag_move = true;
							$('.block').removeClass('click near');
							self.do_move(self._block_line, self._block_column, self._block_line - 1, self._block_column);
						}
					}
				}
			}
		}).on('click', self.elements.start, function() {
			self.start();
			return false;
		}).on('keyup', '.game-username', function() {
			let username = $('.game-username').val();
			self.change_username(username);
		}).on('change', '.game-theme', function() {
			let theme = parseInt($(this).val());
			self.change_theme(theme);
		}).on('touchend mouseup', self.elements.container, function() {
			if (self._start_move) {
				self._start_move = false;
				self._block_element.css('z-index', 10);
				if (!self.drag_move) {
					self.is_clicked(self._block_element);
				}
			}
		}).on('click', self.elements.stop, function() {
			if (self._restart_mode === false) {
				$(this).addClass('attention').html('Are you sure?');
				self._restart_mode = true;
				self.restart_timeout = setTimeout(function () {
					self.ref_el(self.elements.stop).removeClass('attention').html('New Game');
					clearTimeout(self.restart_timeout);
					self._restart_mode = false;
				}, 5000);
			} else {
				self.stop();
			}
			return false;
		}).on('click', '.btn-settings', function() {
			self.ref_el('.panel').hide();
			self.ref_el('.panel.settings').fadeToggle();
			return false;
		}).on('click', '.btn-scores', function() {
			self.ref_el('.scoreboard').empty();
			self.ref_el('.panel').hide();
			self.ref_el('.panel.scores').fadeToggle();
			if (self.highscores.length > 0) {
				self.rank_high_scores();
				for (let i = 0; i < self.highscores.length; i++) {
					self.ref_el('.scoreboard').append('<li><span class="high-score-score">' + self.highscores[i].score + '</span>  <span class="high-score-name">' + self.highscores[i].name + '</span></li>');
					self.ref_el('.loading').hide();
					self.ref_el('.scoreboard').show();
				}
			} else {
				self.ref_el('.loading').html('... no high scores yet ...');
			}
			return false;
		}).on('click', '.btn-about', function() {
			self.ref_el('.panel').hide();
			self.ref_el('.panel.about').fadeToggle();
			return false;
		});
	}

	/*
	 * Update the user name.
	 */
	change_username = function(username) {
		if (username.length > self.MAX_USERNAME_LENGTH) {
			username = username.substring(0, self.MAX_USERNAME_LENGTH);
		}
		this.config.username = username;
		this.set_storage('name', username);
	}

	/*
	 * Update the game theme.
	 */
	change_theme = function(theme) {
		this.set_storage('theme', theme);
		this.config.theme = theme;
	}

	/*
	 * Stop the game.
	 */
	stop = function() {
		this.ref_el(this.elements.stop).removeClass('attention').html('New Game');
		this.reset_ui();
		this.ref_el('.panel').hide();
		this.ref_el('.content').fadeIn();
		this._restart_mode = false;
		clearTimeout(this.restart_timeout);
	}

	/*
	 * Sort the high-scores and keep just the top X.
	 */
	rank_high_scores = function() {
		this.highscores.sort(function(a, b) {
			let keyA = new Date(a.score);
			let keyB = new Date(b.score);
			if (keyA > keyB) {
				return -1;
			}
			if (keyA < keyB) {
				return 1;
			}
			return 0;
		});
		if (this.highscores.length >= this.MAX_HIGHSCORES) {
			this.highscores.splice(this.MAX_HIGHSCORES, this.highscores.length - this.MAX_HIGHSCORES);
		}
	}

	/*
	 * Save the game data.
	 */
	save = function() {
		if (this.scores.points > this.best_scores.points) {
			this.best_scores.points = this.scores.points;
			this.set_storage('bestscore', this.scores.points);
			this.ref_el(this.elements.best_score).html(this.scores.points);
		}
		if (this.scores.combo > this.best_scores.combo) {
			this.best_scores.combo = this.scores.combo;
			this.set_storage('bestcombo', this.scores.combo);
			this.ref_el(this.elements.best_combo).html(this.scores.combo);
		}
		if (this.scores.points > 0) {
			this.highscores.push({
				name: this.config.username,
				score: this.scores.points
			});
		}
		this.set_storage('highscores', JSON.stringify(this.highscores));
	}

	/*
	 * Load initial data from browser localStorage.
	 */
	setup_from_storage = function() {
		let local_best_score = this.get_storage('bestscore');
		if (local_best_score != null) {
			this.best_scores.points = local_best_score;
		}
		this.ref_el(this.elements.best_score).html(this.best_scores.points);
		let local_best_combo = this.get_storage('bestcombo');
		if (local_best_combo != null) {
			this.best_scores.combo = local_best_combo;
		}
		this.ref_el(this.elements.best_combo).html(this.best_scores.combo);
		if (this.get_storage('name') == null) {
			this.ref_el('.panel.settings').show();
		} else {
			this.config.username = this.get_storage('name');
		}
		this.ref_el('.game-username').val(this.config.username);
		if (this.get_storage('theme') == null) {
			this.set_storage('theme', 0);
		}
		this.ref_el('.game-theme').val(this.get_storage('theme'));
		this.config.theme = this.get_storage('theme');
		let local_high_scores = this.get_storage('highscores');
		if (local_high_scores != null) {
			this.highscores = JSON.parse(local_high_scores);
		}
	}

	/*
	 * Save a key with the specified value to the storage.
	 */
	set_storage = function(key, value) {
		localStorage[this.elements.storage + '.' + key] = value;
	}

	/*
	 * Get a key from the storage.
	 */
	get_storage = function(key) {
		return localStorage[this.elements.storage + '.' + key];
	}

	/*
	 * Reset the game data.
	 */
	reset = function() {
		this.scores = {
			points: 0,
			combo: 0
		};
		this.config.blocks = 5;
		this._block_array = [];
		this._block_array_up = [];
		this._test_horizontal = [];
		this._test_vertical = [];
		clearTimeout(this.hint_timeout);
		this._hint_mode = false;
		this.config.multiplier = 0;
		this._block_line = 0;
		this._block_column = 0;
		this.reset_ui();
	}

	/*
	 * Reset the game UI to initial state.
	 */
	reset_ui = function() {
		this.ref_el('.content').hide();
		this.ref_el(this.elements.notification).html('');
		this.ref_el(this.elements.score).html('0');
		this.ref_el(this.elements.combo).html('0');
		this.ref_el(this.elements.moves).html('0');
		this.ref_el('footer').hide();
		this.ref_el('.hint').removeClass('hint');
		this.ref_el(this.elements.moves).removeClass('flash');
	}

	/*
	 * Return a DOM element referenced by the game main DOM element.
	 */
	ref_el = function(element) {
		if (typeof element === 'undefined') {
			return $(this.elements.application);
		} else {
			return $(this.elements.application + ' ' + element);
		}
	}

	/*
	 * Perform a game notification.
	 */
	notify = function(message) {
		this.ref_el(this.elements.notification).html('<div>' + message + '</div>');
	}

	/*
	 * Setup and initialize game board DOM data.
	 */
	setup_board = function() {
		let board_html = '';
		for (let i = 0; i < this.config.lines; i++) {
			this._block_array[i] = [];
			for (let j = 0; j < this.config.columns; j++) {
				let nb_icon = Math.ceil(Math.random() * this.config.blocks);
				if (i > 1) {
					while (this._block_array[i - 2][j] == nb_icon && this._block_array[i - 1][j] == nb_icon) {
						nb_icon = Math.ceil(Math.random() * this.config.blocks);
					}
				}
				if (j > 1) {
					while (this._block_array[i][j - 2] == nb_icon && this._block_array[i][j - 1] == nb_icon) {
						nb_icon = Math.ceil(Math.random() * this.config.blocks);
						if (i > 1) {
							while (this._block_array[i - 2][j] == nb_icon && this._block_array[i - 1][j] == nb_icon) {
									nb_icon = Math.ceil(Math.random() * this.config.blocks);
							}
						}
					}
				}
				this._block_array[i][j] = nb_icon;
				board_html += '<div class="block block_' + nb_icon + '" data-line="' + i + '" data-col="' + j + '" data-block="' + nb_icon + '" style="top: ' + parseInt(i * this.config.size) + 'px; left: ' + parseInt(j * this.config.size) + 'px; width: ' + parseInt(this.config.size) + 'px; height: ' + parseInt(this.config.size) + 'px;"></div>';
			}
		}
		this.ref_el(this.elements.container).html(board_html);
	}

	/*
	 * Start the game.
	 */
	start = function() {
		this.reset();
		if (this.config.username === '') {
			this.config.username = 'user-' + Math.floor(Math.random() * 9999);
			this.set_storage('name', this.config.username);
			this.ref_el('.game-username').val(this.config.username);
		}
		this.resize();
		this.setup_board();
		this.validate_board();
		this.ref_el('footer').show();
	}

	/*
	 * Check if the element got clicked.
	 */
	is_clicked = function(block_element_test) {
		if (!this.ref_el('.block.click').length) {
			block_element_test.addClass('click');
			this.block_test_line = parseInt(block_element_test.attr('data-line'));
			this.block_test_column = parseInt(block_element_test.attr('data-col'));
			this.add_near(this.block_test_line, this.block_test_column);
		} else {
			this.block_reference = this.ref_el('.block.click');
			this.block_reference_line = parseInt(this.block_reference.attr('data-line'));
			this.block_reference_column = parseInt(this.block_reference.attr('data-col'));
			this.block_test_line = parseInt(block_element_test.attr('data-line'));
			this.block_test_column = parseInt(block_element_test.attr('data-col'));
			if ((this.block_reference_line == this.block_test_line && this.block_reference_column == this.block_test_column - 1) || (this.block_reference_line == this.block_test_line && this.block_reference_column == this.block_test_column + 1) || (this.block_reference_line == this.block_test_line - 1 && this.block_reference_column == this.block_test_column) || (this.block_reference_line == this.block_test_line + 1 && this.block_reference_column == this.block_test_column)) {
				this._block_element = this.block_reference;
				this.do_move(this.block_reference_line, this.block_reference_column, this.block_test_line, this.block_test_column);
				this.ref_el('.block').removeClass('click near');
			} else {
				this.ref_el('.block').removeClass('click near');
				block_element_test.addClass('click');
				this.add_near(this.block_test_line, this.block_test_column);
			}
		}
	}

	add_near = function(line, column) {
		if (line > 0) {
			this.ref_el('.block[data-line=' + (line - 1) + '][data-col=' + column + ']').addClass('near');
		}
		if (column > 0) {
			this.ref_el('.block[data-line=' + line + '][data-col=' + (column - 1) + ']').addClass('near');
		}
		if (column < this.config.columns - 1) {
			this.ref_el('.block[data-line=' + line + '][data-col=' + (column + 1) + ']').addClass('near');
		}
		if (line < this.config.lines - 1) {
			this.ref_el('.block[data-line=' + (line + 1) + '][data-col=' + column + ']').addClass('near');
		}
	}

	do_move = function(block_line, block_column, line, column) {
		let self = this;
		this._start_move = false;
		this._forbid_move = true;
		clearTimeout(this.hint_timeout);
		this.ref_el('.hint').removeClass('hint');
		this._hint_mode = false;
		this._selected_element = this.ref_el('.block[data-line=' + line + '][data-col=' + column + ']');
		this._block_element.css('z-index', 10);
		let block_line_origin = block_line;
		let block_col_origin = block_column;
		let block_type_origin = this._block_array[block_line][block_column];
		let selected_line_origin = line;
		let selected_col_origin = column;
		let selected_num_origin = this._block_array[line][column];
		this._block_element.attr('data-line', selected_line_origin);
		this._block_element.attr('data-col', selected_col_origin);
		this._selected_element.attr('data-line', block_line_origin);
		this._selected_element.attr('data-col', block_col_origin);
		this._block_element.css({
			'left': selected_col_origin * self.config.size,
			'top': selected_line_origin * self.config.size
		});
		this._selected_element.css({
			'left': block_col_origin * self.config.size,
			'top': block_line_origin * self.config.size
		});
		this._block_array[block_line_origin][block_col_origin] = selected_num_origin;
		this._block_array[selected_line_origin][selected_col_origin] = block_type_origin;
		setTimeout(function () {
			if (!self.validate_board()) {
				self._block_element.attr('data-line', block_line_origin);
				self._block_element.attr('data-col', block_col_origin);
				self._selected_element.attr('data-line', selected_line_origin);
				self._selected_element.attr('data-col', selected_col_origin);
				self._block_element.css({
					'left': block_col_origin * self.config.size,
					'top': block_line_origin * self.config.size
				});
				self._selected_element.css({
					'left': selected_col_origin * self.config.size,
					'top': selected_line_origin * self.config.size
				});
				self._block_array[block_line_origin][block_col_origin] = block_type_origin;
				self._block_array[selected_line_origin][selected_col_origin] = selected_num_origin;
				setTimeout(function () {
					self.validate_board();
				}, 300);
			}
			self._block_element = undefined;
			self._selected_element = undefined;
		}, 300);
	}

	/*
	 * Perform a full validation of the board.
	 */
	validate_board = function() {
		let self = this;
		for (let i = 0; i < this.config.lines; i++) {
			this._block_array_up[i] = [];
			this._test_vertical[i] = [];
			this._test_horizontal[i] = [];
			for (let j = 0; j < this.config.columns; j++) {
				this._block_array_up[i][j] = false;
				this._test_horizontal[i][j] = false;
				this._test_vertical[i][j] = false;
			}
		}
		this.ref_el('.block.blackhole').removeClass('new');
		let chain_found = false;
		for (let i = 0; i < this.config.lines; i++) {
			for (let j = 0; j < this.config.columns; j++) {
				if (this.is_chain(i, j)) {
					chain_found = true;
				}
			}
		}
		if (self._block_element != undefined && self._selected_element != undefined) {
			if (self._block_element.hasClass('blackhole') && !self._block_element.hasClass('new')) {
				self.remove_similar_blocks(self._selected_element.attr('data-block'), self._block_element.attr('data-line'), self._block_element.attr('data-col'));
				self._block_array_up[self._block_element.attr('data-line')][self._block_element.attr('data-col')] = true;
				chain_found = true;
				self.config.multiplier++;
				if (self.config.multiplier > self.scores.combo) {
					self.scores.combo = self.config.multiplier;
					self.ref_el(self.elements.combo).html(self.scores.combo);
				}
			}
			if (self._selected_element.hasClass('blackhole') && !self._selected_element.hasClass('new')) {
				self.remove_similar_blocks(self._block_element.attr('data-block'), self._selected_element.attr('data-line'), self._selected_element.attr('data-col'));
				self._block_array_up[self._selected_element.attr('data-line')][self._selected_element.attr('data-col')] = true;
				chain_found = true;
				self.config.multiplier++;
				if (self.config.multiplier > self.scores.combo) {
					self.scores.combo = self.config.multiplier;
					self.ref_el(self.elements.combo).html(self.scores.combo);
				}
			}
		}
		if (chain_found) {
			clearTimeout(self.lightspeed_timeout);
			for (let i = 0; i < self.config.lines; i++) {
				for (let j = 0; j < self.config.columns; j++) {
					if (self._block_array_up[i][j]) {
						self._block_array[i][j] = 0;
						self.ref_el('.block[data-line=' + i + '][data-col=' + j + ']').fadeOut(300, function () {
							$(this).remove();
						});
						self.scores.points += (self.SCORE_MULTIPLIER * self.config.multiplier);
					}
				}
			}
			self.ref_el(self.elements.score).html(self.scores.points);
			setTimeout(function () {
				self.fall_blocks();
				setTimeout(function () {
					self.validate_board();
				}, 400);
			}, 400);
		} else {
			if (self._block_element == undefined && self._selected_element == undefined) {
				if (self.is_move()) {
					self._forbid_move = false;
					if (self.scores.points >= 500 && self.scores.points < 1000) {
						self.config.blocks = 6;
					}
					else if (self.scores.points >= 1000 && self.scores.points < 1500) {
						self.config.blocks = 7;
					}
					else if (self.scores.points >= 1500 && self.scores.points < 2000) {
						self.config.blocks = 8;
					}
					else if (self.scores.points >= 2000 && self.scores.points < 2500) {
						self.config.blocks = 9;
					}
					else if (self.scores.points >= 2500 && self.scores.points < 3000) {
						self.config.blocks = 10;
					}
					else if (self.scores.points >= 3000 && self.scores.points < 3500) {
						self.config.blocks = 11;
					}
					else if (self.scores.points >= 3500) {
						self.config.blocks = 12;
					} else {
						self.config.blocks = 5;
					}
					self.lightspeed_timeout = setTimeout(function () {
						self.config.multiplier = 0;
					}, self.TIMEOUT_LIGHTSPEED);
					self.hint_timeout = setTimeout(function () {
						self._hint_mode = true;
						self.is_move();
					}, self.TIMEOUT_HINT);
				} else {
					self.notify('GAME OVER<br />Score: ' + self.scores.points + '<br />Max combo: ' + self.scores.combo);
					self.save();
				}
			}
		}
		return chain_found;
	}

	is_chain = function(line, column) {
		let self = this;
		let chain_found = false;
		let block_type = this._block_array[line][column];
		let chain_vertical = 1;
		let chain_horizontal = 1;
		let i;
		if (!this._test_vertical[line][column]) {
			i = 1;
			while (((line + i) < this.config.lines) && (this._block_array[line + i][column] == block_type) && (!this._test_vertical[line + i][column])) {
				chain_vertical++;
				i++;
			}
			if (chain_vertical >= 3) {
				chain_found = true;
				this.config.multiplier++;
				if (this.config.multiplier > this.scores.combo) {
					this.scores.combo = this.config.multiplier;
					this.ref_el(this.elements.combo).html(this.scores.combo);
				}
				if (self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').hasClass('supernova')) {
					self.remove_blocks(line, column);
				}
				if (self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').hasClass('gravitron')) {
					self.remove_line_column(line, column);
				}
				self._block_array_up[line][column] = true;
				self._test_vertical[line][column] = true;
				if (chain_vertical == 4) {
					self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').css({
						'top': (line + 1) * self.config.size
					});
				}
				if (chain_vertical == 5) {
					self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').css({
						'top': (line + 2) * self.config.size
					});
				}
				i = 1;
				while (line + i < self.config.lines && self._block_array[line + i][column] == block_type) {
					if (self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').hasClass('supernova')) {
						self.remove_blocks(line + i, column);
					}
					if (self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').hasClass('gravitron')) {
						self.remove_line_column(line + i, column);
					}
					if (chain_vertical == 4) {
						self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').css({
							'top': (line + 1) * self.config.size
						});
					}
					if (chain_vertical == 5) {
						self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').css({
							'top': (line + 2) * self.config.size
						});
					}
					if (i == 1 && self.config.multiplier % 5 == 0) {
						self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').addClass('gravitron');
					} else {
						if (i == 1 && chain_vertical == 4) {
							self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').addClass('supernova');
						} else {
							if (i == 2 && chain_vertical == 5) {
								self.ref_el('.block[data-line=' + (line + i) + '][data-col=' + column + ']').removeClass('block_1 block_2 block_3 block_4 block_5 block_6 block_7 block_8 block_9 block_10 block_11 block_12').addClass('blackhole new');
								self._block_array[line + i][column] = 13;
							} else {
								self._block_array_up[line + i][column] = true;
								self._test_vertical[line + i][column] = true;
							}
						}
					}
					i++;
				}
			}
		}
		if (!self._test_horizontal[line][column]) {
			i = 1;
			while (column + i < self.config.columns && self._block_array[line][column + i] == block_type && !self._test_horizontal[line][column + i]) {
				chain_horizontal++;
				i++;
			}
			if (chain_horizontal >= 3) {
				chain_found = true;
				self.config.multiplier++;
				if (self.config.multiplier > self.scores.combo) {
					self.scores.combo = self.config.multiplier;
					self.ref_el(self.elements.combo).html(self.scores.combo);
				}
				if (self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').hasClass('supernova')) {
					self.remove_blocks(line, column);
				}
				if (self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').hasClass('gravitron')) {
					self.remove_line_column(line, column);
				}
				self._block_array_up[line][column] = true;
				self._test_horizontal[line][column] = true;
				if (chain_horizontal == 4) {
					self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').css({
						'left': (column + 1) * self.config.size
					});
				}
				if (chain_horizontal == 5) {
					self.ref_el('.block[data-line=' + line + '][data-col=' + column + ']').css({
						'left': (column + 2) * self.config.size
					});
				}
				i = 1;
				while (column + i < self.config.columns && self._block_array[line][column + i] == block_type) {
					if (self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').hasClass('supernova')) {
						self.remove_blocks(line, column + i);
					}
					if (self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').hasClass('gravitron')) {
						self.remove_line_column(line, column + i);
					}
					if (chain_horizontal == 4) {
						self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').css({
							'left': (column + 1) * self.config.size
						});
					}
					if (chain_horizontal == 5) {
						self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').css({
							'left': (column + 2) * self.config.size
						});
					}
					if (i == 1 && self.config.multiplier % 5 == 0) {
						self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').addClass('gravitron');
					} else {
						if (i == 1 && chain_horizontal == 4) {
							self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').addClass('supernova');
						} else {
							if (i == 2 && chain_horizontal == 5) {
								self.ref_el('.block[data-line=' + line + '][data-col=' + (column + i) + ']').removeClass('block_1 block_2 block_3 block_4 block_5 block_6 block_7 block_8 block_9 block_10 block_11 block_12').addClass('blackhole new');
								self._block_array[line][column + i] = 13;
							} else {
								self._block_array_up[line][column + i] = true;
								self._test_horizontal[line][column + i] = true;
							}
						}
					}
					i++;
				}
			}
		}
		return chain_found;
	}

	remove_blocks = function(line, column) {
		if (line > 0 && column > 0) {
			this._block_array_up[line - 1][column - 1] = true;
		}
		if (line > 0) {
			this._block_array_up[line - 1][column] = true;
		}
		if (line > 0 && column < this.config.columns - 1) {
			this._block_array_up[line - 1][column + 1] = true;
		}
		if (column > 0) {
			this._block_array_up[line][column - 1] = true;
		}
		if (column < this.config.columns - 1) {
			this._block_array_up[line][column + 1] = true;
		}
		if (line < this.config.lines - 1 && column > 0) {
			this._block_array_up[line + 1][column - 1] = true;
		}
		if (line < this.config.lines - 1) {
			this._block_array_up[line + 1][column] = true;
		}
		if (line < this.config.lines - 1 && column < this.config.columns - 1) {
			this._block_array_up[line + 1][column + 1] = true;
		}
		this.explode(line, column);
	}

	/*
	 * Explode the specified element(s).
	 */
	explode = function(line, column) {
		let self = this;
		let explode_element = $('<div class="explosion"></div>');
		explode_element.css({
			'left': (column - 1) * self.config.size,
			'top': (line - 1) * self.config.size,
			'width': self.config.size + 'px',
			'height': self.config.size + 'px'
		});
		this.ref_el(this.elements.container).append(explode_element);
		setTimeout(function () {
			explode_element.remove();
		}, 600);
	}

	/*
	 * Remove all blocks similar to the specified one.
	 */
	remove_similar_blocks = function(block_type, line, column) {
		let self = this;
		for (let i = 0; i < this.config.lines; i++) {
			for (let j = 0; j < this.config.columns; j++) {
				if (this._block_array[i][j] == block_type) {
					this._block_array_up[i][j] = true;
					this.ref_el('.block[data-line=' + i + '][data-col=' + j + ']').css({
						'left': column * self.config.size,
						'top': line * self.config.size,
						'width': self.config.size + 'px',
						'height': self.config.size + 'px'
					});
				}
			}
		}
	}

	remove_line_column = function(line, column) {
		let self = this;
		for (let i = 0; i < this.config.lines; i++) {
			this._block_array_up[i][column] = true;
			this.ref_el('.block[data-line=' + i + '][data-col=' + column + ']').css({
				'left': column * self.config.size,
				'top': line * self.config.size,
				'width': self.config.size + 'px',
				'height': self.config.size + 'px'
			});
		}
		for (let i = 0; i < this.config.lines; i++) {
			this._block_array_up[line][i] = true;
			this.ref_el('.block[data-line=' + line + '][data-col=' + i + ']').css({
				'left': column * self.config.size,
				'top': line * self.config.size,
				'width': self.config.size + 'px',
				'height': self.config.size + 'px'
			});
		}
	}

	fall_blocks = function() {
		let self = this;
		for (let i = this.config.lines - 1; i >= 0; i--) {
			for (let j = 0; j < this.config.columns; j++) {
				if (this._block_array[i][j] == 0) {
					let k = 1;
					while ((i - k) >= 0 && this._block_array[i - k][j] == 0) {
						k++;
					}
					if ((i - k) < 0) {
						let random_block = Math.ceil(Math.random() * self.config.blocks);
						let new_block = $('<div class="block block_' + random_block + '" data-line="' + i + '" data-col="' + j + '" data-block="' + random_block + '"></div>');
						new_block.css({
							'left': j * self.config.size,
							'top': -self.config.size,
							'width': self.config.size + 'px',
							'height': self.config.size + 'px'
						});
						self.ref_el(self.elements.container).append(new_block);
						new_block.animate({
							'top': i * self.config.size
						}, 0);
						self._block_array[i][j] = random_block;
					} else {
						let fall_block = self.ref_el('.block[data-line=' + (i - k) + '][data-col=' + j + ']');
						fall_block.attr('data-line', i);
						fall_block.css('top', i * self.config.size);
						self._block_array[i][j] = self._block_array[i - k][j];
						self._block_array[i - k][j] = 0;
					}
				}
			}
		}
	}

	is_move = function() {
		let self = this;
		let found_move = false;
		let show_hint = false;
		let possible_moves = 0;
		for (let i = 0; i < this.config.lines; i++) {
			this._test_tab[i] = [];
			for (let j = 0; j < this.config.columns; j++) {
				this._test_tab[i][j] = this._block_array[i][j];
			}
		}
		for (let i = 0; i < self.config.lines; i++) {
			for (let j = 0; j < self.config.columns; j++) {
				if (j < self.config.columns - 1) {
					self._test_tab[i][j] = self._block_array[i][j + 1];
					self._test_tab[i][j + 1] = self._block_array[i][j];
					if (self.is_over(i, j)) {
						found_move = true;
						possible_moves++;
						if (self._hint_mode && !show_hint) {
							self.ref_el('.block[data-line=' + i + '][data-col=' + (j + 1) + ']').addClass('hint');
							show_hint = true;
						}
					}
					if (self.is_over(i, j + 1)) {
						found_move = true;
						possible_moves++;
						if (self._hint_mode && !show_hint) {
							self.ref_el('.block[data-line=' + i + '][data-col=' + j + ']').addClass('hint');
							show_hint = true;
						}
					}
					self._test_tab[i][j] = self._block_array[i][j];
					self._test_tab[i][j + 1] = self._block_array[i][j + 1];
				}
				if (i < self.config.lines - 1) {
					self._test_tab[i][j] = self._block_array[i + 1][j];
					self._test_tab[i + 1][j] = self._block_array[i][j];
					if (self.is_over(i, j)) {
						found_move = true;
						possible_moves++;
						if (self._hint_mode && !show_hint) {
							self.ref_el('.block[data-line=' + (i + 1) + '][data-col=' + j + ']').addClass('hint');
							show_hint = true;
						}
					}
					if (self.is_over(i + 1, j)) {
						found_move = true;
						possible_moves++;
						if (self._hint_mode && !show_hint) {
							self.ref_el('.block[data-line=' + i + '][data-col=' + j + ']').addClass('hint');
							show_hint = true;
						}
					}
					self._test_tab[i][j] = self._block_array[i][j];
					self._test_tab[i + 1][j] = self._block_array[i + 1][j];
				}
			}
		}
		if (possible_moves <= 3) {
			this.ref_el(this.elements.moves).addClass('flash').html(possible_moves);
		} else {
			this.ref_el(this.elements.moves).removeClass('flash').html(possible_moves);
		}
		return found_move;
	}

	/*
	 * Check if game is over.
	 */
	is_over = function(line, column) {
		let chain_found = false;
		let block_type = this._test_tab[line][column];
		let chain_vertical = 1;
		let chain_horizontal = 1;
		let i = 1;
		while (line - i >= 0 && this._test_tab[line - i][column] == block_type) {
			chain_vertical++;
			i++;
		}
		i = 1;
		while (line + i < this.config.lines && this._test_tab[line + i][column] == block_type) {
			chain_vertical++;
			i++;
		}
		i = 1;
		while (column - i >= 0 && this._test_tab[line][column - i] == block_type) {
			chain_horizontal++;
			i++;
		}
		i = 1;
		while (column + i < this.config.lines && this._test_tab[line][column + i] == block_type) {
			chain_horizontal++;
			i++;
		}
		if (chain_vertical >= 3) {
			chain_found = true;
		}
		if (chain_horizontal >= 3) {
			chain_found = true;
		}
		if (this._test_tab[line][column] == 13) {
			chain_found = true;
		}
		return chain_found;
	}

	/*
	 * Resize the game board DOM elements.
	 */
	resize = function(force) {
		this.ref_el(this.elements.container).height(this.ref_el().height() - this.ref_el('footer').height() - this.ref_el('header').height() - 20);
		this.config.columns = Math.floor(this.ref_el(this.elements.container).width() / this.config.size);
		this.config.lines = Math.floor(this.ref_el(this.elements.container).height() / this.config.size);
		this.ref_el(this.elements.container).width(this.config.columns * this.config.size);
		this.ref_el(this.elements.container).attr('class', 'theme' + this.config.theme);
		if (force === true) {
			this.resize_board();
		}
	}

	resize_board = function() {
		let self = this;
		for (let i = 0; i < this.config.lines; i++) {
			for (let j = 0; j < this.config.columns; j++) {
				this.ref_el('.block[data-line=' + i + '][data-col=' + j + ']').css({
					'left': j * self.config.size + 'px',
					'top': i * self.config.size + 'px',
					'width': self.config.size + 'px',
					'height': self.config.size + 'px'
				});
			}
		}
	}
}

$(document).ready(function() {
	new game();
});
