unit GPSparser;

{$mode ObjFPC}{$H+}

interface

uses
	Classes, SysUtils;

type
	EGPSError = class(Exception);

	TGPSDecimalInfo = record
		Latitude: Extended;
		Longitude: Extended;
	end;

	TGPSInfo = record
		Source: String;
        Latitude: String;
        Longitude: String;
		Decimal: TGPSDecimalInfo;
		LastAccess: TDateTime;
        Fix: String;
        Satellites: String;
        HDOP: String;
        Altitude: String;
	end;

function GPSDecimalToDMS(Value: Extended; D, M, S: String): String; overload;
function GPSDecimalToDMS(Value: Extended): String; overload;
function GPSDecimalToDM(Value: Extended; D, M: String): String; overload;
function GPSDecimalToDM(Value: Extended): String; overload;
function GPSToMAP(Info: TGPSDecimalInfo): String; overload;
function GPSToMAP(Latitude: Extended; Longitude: Extended): String; overload;
function GPSParse(Command: String; var Info: TGPSInfo): Boolean;

implementation

function GPSDecimalToDMS(Value: Extended; D, M, S: String): String;
var
	t: Integer;
begin
	t := Trunc(Value);
	Result := IntToStr(t) + ' ';
	Value := (Value - t) * 60;
	t := Trunc(Value);
	Result := Result + IntToStr(t) + ' ';
	Value := (Value - t) * 60;
	Result := Result + FormatFloat('0.0000', Value);
end;

function GPSDecimalToDMS(Value: Extended): String;
begin
	Result := GPSDecimalToDMS(Value, ' ', ' ', '');
end;

function GPSDecimalToDM(Value: Extended; D, M: String): String;
var
	t: Integer;
begin
	t := Trunc(Value);
	Result := IntToStr(t) + D;
	Value := (Value - t) * 60;
	Result := Result + FormatFloat('0.0000', Value) + M;
end;

function GPSDecimalToDM(Value: Extended): String;
begin
	Result := GPSDecimalToDM(Value, ' ', '');
end;

function GPSToMAP(Latitude: Extended; Longitude: Extended): String; overload;
begin
  if Latitude < 0 then
    Result := '-'
  else
    Result := '+';
  Result := Result + FormatFloat('0.000000', Abs(Latitude));
  Result := Result + ',';
  if Longitude < 0 then
    Result := Result + '-'
  else
    Result := Result + '+';
  Result := Result + FormatFloat('0.000000', Abs(Longitude));
end;

function GPSToMAP(Info: TGPSDecimalInfo): String;
begin
	Result := GPSToMAP(Info.Latitude, Info.Longitude)
end;

function GPSParse(Command: String; var Info: TGPSInfo): Boolean;
var
	P: Integer;
	CS: String;
	D, M: string;
	Params: TStringList;
	CMD: String;
begin
	Result := False;
	Command := Trim(Command);
	if LeftStr(Command, 3) = '$GP' then //is it GPS command?
	begin
		Command := Copy(Command, 4, MaxInt);//remove '$GP'
		P := AnsiPos('*', Command);
		if P > 0 then // not bad for now
		begin
			CS := Copy(Command, P + 1, MaxInt);
			Command := Copy(Command, 1, P - 1);
			Params := TStringList.Create;
			try
				ExtractStrings([','], [], PChar(Command), Params);
				CMD := Params[0];
				if CMD = 'GSV' then
                	// Satellite info
				else if CMD = 'GSA' then
                	// Fix
				else if CMD = 'GGA' then
				begin
					Result := True;
					Info.Source := Params[3] + Params[2] + ',' + Params[5] + Params[4];
                    Info.Latitude := Params[2] + ' ' + Params[3];
                    Info.Longitude := Params[4] + ' ' + Params[5];
					D := Copy(Params[2], 1, 2); //3 char for Degree
					M := Copy(Params[2], 3, MaxInt);
					Info.Decimal.Latitude := StrToIntDef(D, 0) + (StrToFloatDef(M, 0) / 60);
					if Params[3] = 'S' then
						Info.Decimal.Latitude := -Info.Decimal.Latitude;
					D := Copy(Params[4], 1, 3); //3 char for Degree
					M := Copy(Params[4], 4, MaxInt);
					Info.Decimal.Longitude := StrToIntDef(D, 0) + (StrToFloatDef(M, 0) / 60);
					if Params[5] = 'W' then
						Info.Decimal.Longitude := -Info.Decimal.Longitude;
					Info.LastAccess := Now;
                    Info.Fix := Params[6];
                    Info.Satellites := Params[7];
                    Info.HDOP := Params[8];
                    Info.Altitude := Params[9] + ' ' + Params[10];
				end;
			finally
				Params.Free;
			end;
		end;
	end;
end;

end.

