# Biome world generation library

Generates a world based on elevation and moisture, complete with terrain features and exports the SVG image and the JSON world data object.

## Usage

Create the main world object based on the provided settings and populate the internal array. All parameters are optional:

```js
let new_world = new world({
	width: 64,				/* world width, in hexes */
	height: 32,				/* world height, in hexes */
	erosion: 0.54,			/* world erosion */
	seed_moisture: 6893		/* Load a specific random seed */
	seed_elevation: 736		/* Load a specific random seed */
});
```

Draw the internal JSON world data as an image:

```html
<div class="worldmap"></div>
```

```js
/* draw to a SVG image with 0.5 terrain opacity,  all other parameters are optional */
new_world.draw('.worldmap', {
	mode: 'svg',			/* map mode, SVG or HTML5 canvas */
	opacity: 0.5,			/* terrain opacity */
	hex_size: 24,			/* size of a single hex */
	show_grid: true,		/* whether to display the hex grid */
	show_terrain: true,		/* whether to display the advanced terrain features */
	assets_url: './'		/* Path to load the terrain assets PNGs from */
});

/* draw to a HTML5 canvas with 0.4 terrain opacity */
new_world.draw('.worldmap', {
	mode: 'canvas',
	opacity: 0.4
});

/* draw to a HTML5 canvas with default options */
new_world.draw('.worldmap', {
	mode: 'canvas'
});
```

Get the properties of the world:

```js
let properties = new_world.props();

/* will result in */

{
	width: 64,
	height: 32,
	erosion: 0.54,
	seeds: {
		elevation: 736,
		moisture: 6893
	}
}
```

Export the internal JSON world data:

```js
/* export JSON serialized data */
let exported_data = new_world.export_json();

/* export raw Array object */
let exported_data = new_world.export();
```

Manipulate the world data array:

```js
let world_data = new_world.data();
let properties = new_world.props();
for (let y = 0; y < properties.height; y++) {
	for (let x = 0; x < properties.width; x++) {
		console.log(world_data[y][x].e);
	}
}
```

Get the default terrain colors:

```js
let terrains = new_world.terrains();

/* will result in */

terrains['grass'].color = '#E6F59A';

/* and you can get the color of a hex too: */
let terrain = new_world.get_hex_terrain({
	x: 12,
	y: 26
});
let terrains = new_world.terrains();
let hex_color = terrains[terrain].color;

/* if terrain = 'mountains', hex_color will be: */

hex_color = '#B37D1A';
```

Check if a hex is water:

```js
let is_water = new_world.hex_is_water({
	x: 12,
	y: 26
});

/* will result in */

is_water = false;
```

Get the terrain type of the specified hex, based on its elevation:

```js
let terrain = new_world.get_hex_terrain({
	x: 12,
	y: 26
});

/* will result in */

terrain = 'mountains';
```

Get the current elevation of a hex:

```js
let elevation = new_world.get_hex_elevation({
	x: 2,
	y: 4
});

/* will result in */

elevation = 0.7391983661686046;
```

Get the current moisture of a hex:

```js
let moisture = new_world.get_hex_moisture({
	x: 10,
	y: 11
});

/* will result in */

moisture = 0.4925356581346529;
```

Get a random hex from the world data with the specified terrain (or `false` if there is no hex with that terrain type).

```js
let hex = new_world.get_random_hex_with_terrain('grass');

/* will result in */

hex = {
	x: 10,
	y: 22
}
```

Get terrain statistics (number of hexes for each terrain type).

```js
let stats = new_world.stats();

/* will result in */

stats = { ocean: 0, grass: 2086, subtropical_desert: 11, temperate_desert: 12, taiga: 0, ... }
```

Raise the current moisture of a hex by 0.12:

```js
let new_moisture = new_world.raise_hex_moisture({
	x: 10,
	y: 11
}, 0.12);

/* will result in */

new_moisture = 0.4925356581346529 + 0.12;

/* if you want to lower the moisture, use the lower_hex_moisture() method */
```

Lower the current elevation of a hex by 0.87:

```js
let new_elevation = new_world.lower_hex_elevation({
	x: 10,
	y: 11
}, 0.87);

/* will result in */

new_elevation = 0.7391983661686046 - 0.87;

/* if you want to raise the elevation, use the raise_hex_elevation() method */
```

Get the world data for the specified hex.

```js
let hex_data = new_world.get_hex({
	x: 4,
	y: 2
});

/* will result in */

hex_data = {
	e: 0.7391983661686046,
	m: 0.6185405664699859
}
```

Set a hex data key to the specified value. You can use this method to store additional information into the world data array.

```js
new_world.set_hex(hex, 'city', {
	name: 'Alexandria',
	population: 130000
});
```

Load raw world data into the generator:

```js
let world_data = [[{"e":0.7578606610586678,"m":0.5642837814565862},{"e":0.7902203979081555,"m":...
new_world.data(world_data);
```

## License

This library is written by __sizeof(cat)__ and released under __GPLv3__.

```txt
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program. If not, see <https://www.gnu.org/licenses/>.
```

## Acknowledgements

- Uses the [Simplex noise](https://github.com/jwagner/simplex-noise.js) Javascript implementation.[^1]
- Uses a [Pseudo-Random Number Generator](https://github.com/odogono/prng-parkmiller-js) for javascript.[^2]
- Uses the [canvg library](https://github.com/canvg/canvg) that converts a SVG to a PNG.[^3]
- Uses the terrain tiles that were created for [Hexographer](http://www.hexographer.com/) by Inkwell Ideas, Inc.[^4]

## Downloads

- Medium (128x128) archipelago map, __The Caribbean__ - [json](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/archipelago.json.zip), [png](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/archipelago.png) (1.7MB), [moisture map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/archipelago-moisture-map.png)/[elevation map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/archipelago-elevation-map.png)
- Medium (128x128) mountain pass map, __Cordillera de los Andes__ - [json](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/mountain-pass.json.zip), [png](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/mountain-pass.png) (1.9MB), [moisture map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/mountain-pass-moisture-map.png)/[elevation map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/mountain-pass-elevation-map.png)
- Medium (128x128) inland lake map, __Crater Lake__ - [json](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-lake.json.zip), [png](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-lake.png) (2.1MB), [moisture map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-lake-moisture-map.png)/[elevation map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-lake-elevation-map.png)
- Medium (128x128) inland sea map, __Lush__ - [json](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-sea.json.zip), [png](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/inland-sea.png) (2MB), [moisture map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island-sea-moisture-map.png)/[elevation map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island-sea-elevation-map.png)
- Medium (128x128) island map, __Kahuna__ - [json](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island.json.zip), [png](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island.png) (1.1MB), [moisture map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island-moisture-map.png)/[elevation map](https://sizeof.cat/project/software/tools/hex-world-generator/files/maps/island-elevation-map.png)

## Screenshots

| | | | |
|---|---|---|---|
| ![](img/screenshots/hex-world-gen1.png) | ![](img/screenshots/hex-world-gen2.png) | ![](img/screenshots/hex-world-gen3.png) | ![](img/screenshots/hex-world-gen4.png) |

[^1]: Copyright (c) 2015 Jonas Wagner, licensed under the MIT license.
[^2]: Copyright (c) 2009 Michael Baczynski, [polygonal.de](http://www.polygonal.de), ported by Alex Veenendaal, [opendoorgonorth.com](http://opendoorgonorth.com), licensed under the MIT license.
[^3]: Copyright (c) 2010-2011 Gabe Lerner, [code.google.com/p/canvg](http://code.google.com/p/canvg/), licensed under the MIT license.
[^4]: Released to the public domain by Inkwell Ideas, Inc.

## More info

- [Perlin noise](https://en.wikipedia.org/wiki/Perlin_noise), [Simplex noise](https://en.wikipedia.org/wiki/Simplex_noise) and [OpenSimplex noise](https://en.wikipedia.org/wiki/OpenSimplex_noise)
- [Making maps with noise functions](https://www.redblobgames.com/maps/terrain-from-noise)
- [Hexagonal Grids](https://www.redblobgames.com/grids/hexagons/)
- [Perlin Noise, Procedural Content Generation, and Interesting Space](https://heredragonsabound.blogspot.com/2019/02/perlin-noise-procedural-content.html)
- [Geometry with Hex Coordinates](http://devmag.org.za/2013/08/31/geometry-with-hex-coordinates/)
- [r/proceduralgeneration](https://old.reddit.com/r/proceduralgeneration/)
