/*!
 * Cupcakes game.
 *
 * @author sizeof(cat) <sizeofcat AT riseup.net>
 * @copyright 2020 - 2023 sizeof(cat)
 * @version 2.2.0
 * @license GPLv3 https://sizeof.cat/LICENSE.txt
 * 
 *           o8o                                  .o88o.   .o                         .   o.   
 *           `"'                                  888 `"  .8'                       .o8   `8.  
 *  .oooo.o oooo    oooooooo  .ooooo.   .ooooo.  o888oo  .8'   .ooooo.   .oooo.   .o888oo  `8. 
 * d88(  "8 `888   d'""7d8P  d88' `88b d88' `88b  888    88   d88' `"Y8 `P  )88b    888     88 
 * `"Y88b.   888     .d8P'   888ooo888 888   888  888    88   888        .oP"888    888     88 
 * o.  )88b  888   .d8P'  .P 888    .o 888   888  888    `8.  888   .o8 d8(  888    888 .  .8' 
 * 8""888P' o888o d8888888P  `Y8bod8P' `Y8bod8P' o888o    `8. `Y8bod8P' `Y888""8o   "888" .8'  
 *                                                         `"                             "'   
 */

/*
 * Main game object.
 */
class game {
	/*
	 * Object constructor, gets ran on object initialization.
	 */
	 constructor (params) {

		/*
		 * Current scores.
		 */
		this.scores = {
			points: 0,
			combo: 0
		};

		/*
		 * Best scores.
		 */
		this.best_scores = {
			points: 0,
			combo: 0
		};

		/*
		 * Game configuration.
		 */
		this.config = {
			lines: 12,
			columns: 11,
			theme: 0,
			username: '',
			elements: {
				width: 48,
				height: 48
			}
		};

		/*
		 * Game DOM elements.
		 */
		this.elements = {
			application: '.application',
			container: '.game-container',
			notification: '.notification',
			start: '.btn-start',
			stop: '.btn-stop',
			best_score: '.best_score',
			score: '.score',
			best_combo: '.best_combo',
			combo: '.combo',
			storage: 'cupcakes'
		};

		/*
		 * List of game themes.
		 */
		this.themes = [
			{
				name: 'default'
			}, {
				name: 'fruits'
			}
		];

		/*
		 * Array containing the high scores list.
		 */
		this.highscores = [];

		/*
		 * Number of high scores to show in the list.
		 */
		this.MAX_HIGHSCORES = 3;

		/*
		 * Don't allow an username to be longer than this.
		 */
		this.MAX_USERNAME_LENGTH = 16;

		/*
		 * Game version.
		 */
		this.VERSION = '2.2.0';

		/*
		 * Various internal timeouts.
		 */
		this.restart_timeout = null;

		/*
		 * Private variables.
		 */
		this._level;
		this._combo_best_current;
		this._score_per_level;
		this._cycle;
		this._evil;
		this._colors;
		this._points;
		this._restart_mode = false;

		this.setup_from_storage();
		this.build_ui();
		this.attach_events();
		this.init();
	}

	build_ui () {
		this.ref_el('.game-container').height(this.ref_el().height() - this.ref_el('footer').height() - this.ref_el('header').height() - 20);
		this.config.lines = Math.floor(this.ref_el('.game-container').width() / this.config.elements.width);
		this.config.columns = Math.floor(this.ref_el('.game-container').height() / this.config.elements.height);
		this.ref_el('.game-username').attr('maxlength', this.MAX_USERNAME_LENGTH);
		for (let i = 0; i < this.themes.length; i++) {
			let id = i;
			if (this.themes[i].id !== undefined) {
				id = this.themes[i].id;
			}
			this.ref_el('.game-theme').append('<option value="' + id + '">' + this.themes[i].name + '</option>');
		}
		this.ref_el('.game-version').html('v' + this.VERSION);
	}

	attach_events () {
		let self = this;
		this.ref_el().on('click', this.elements.start, function () {
			self.restart();
			return false;
		}).on('click', '.btn-restart', function () {
			$('.application .panel').hide();
			$('.application .content').fadeIn();
			return false;
		}).on('click', this.elements.stop, function() {
			if (self._restart_mode === false) {
				$(this).addClass('attention').html('Are you sure?');
				self._restart_mode = true;
				self.restart_timeout = setTimeout(function () {
					self.ref_el(self.elements.stop).removeClass('attention').html('New Game');
					clearTimeout(self.restart_timeout);
					self._restart_mode = false;
				}, 5000);
			} else {
				self.stop();
			}
			return false;
		}).on('click', '.btn-settings', function () {
			$('.application .panel').hide();
			$('.application .panel.settings').fadeToggle();
		}).on('click', '.btn-scores', function () {
			self.ref_el('.scoreboard').empty();
			self.ref_el('.panel').hide();
			self.ref_el('.panel.scores').fadeToggle();
			if (self.highscores.length > 0) {
				self.rank_high_scores();
				for (let i = 0; i < self.highscores.length; i++) {
					self.ref_el('.scoreboard').append('<li><span class="high-score-score">' + self.highscores[i].score + '</span>  <span class="high-score-name">' + self.highscores[i].name + '</span></li>');
					self.ref_el('.loading').hide();
					self.ref_el('.scoreboard').show();
				}
			} else {
				self.ref_el('.loading').html('... no high scores yet ...');
			}
			return false;
		}).on('click', '.btn-about', function () {
			$('.application .panel').hide();
			$('.application .panel.about').fadeToggle();
		}).on('change', '.game-theme', function() {
			let theme = parseInt($(this).val());
			self.change_theme(theme);
		}).on('keyup', '.game-username', function() {
			let username = $('.game-username').val();
			self.change_username(username);
		});
		if (document.body.addEventListener) {
			document.body.addEventListener("touchmove", function (e) {
				e.preventDefault();
			}, false);
		}
	}

	/*
	 * Sort the highscores and keep just the top X.
	 */
	rank_high_scores () {
		this.highscores.sort(function(a, b) {
			let keyA = new Date(a.score);
			let keyB = new Date(b.score);
			if (keyA > keyB) {
				return -1;
			}
			if (keyA < keyB) {
				return 1;
			}
			return 0;
		});
		if (this.highscores.length >= this.MAX_HIGHSCORES) {
			this.highscores.splice(this.MAX_HIGHSCORES, this.highscores.length - this.MAX_HIGHSCORES);
		}
	}

	/*
	 * Save the game data.
	 */
	save () {
		if (this.scores.points > this.best_scores.points) {
			this.best_scores.points = this.scores.points;
			this.set_storage('bestscore', this.scores.points);
			this.ref_el(this.elements.best_score).html(this.scores.points);
		}
		if (this.scores.combo > this.best_scores.combo) {
			this.best_scores.combo = this.scores.combo;
			this.set_storage('bestcombo', this.scores.combo);
			this.ref_el(this.elements.best_combo).html(this.scores.combo);
		}
		if (this.scores.points > 0) {
			this.highscores.push({
				name: this.config.username,
				score: this.scores.points
			});
		}
		this.set_storage('highscores', JSON.stringify(this.highscores));
	}

	/*
	 * Update the game theme.
	 */
	change_theme (theme) {
		this.set_storage('theme', theme);
		this.config.theme = theme;
	}

	/*
	 * Update the user name.
	 */
	change_username (username) {
		if (username.length > this.MAX_USERNAME_LENGTH) {
			username = username.substring(0, this.MAX_USERNAME_LENGTH);
		}
		this.config.username = username;
		this.set_storage('name', username);
	}

	/*
	 * Load initial data from browser localStorage.
	 */
	setup_from_storage () {
		let local_best_score = this.get_storage('bestscore');
		if (local_best_score != null) {
			this.best_scores.points = local_best_score;
		}
		this.ref_el(this.elements.best_score).html(this.best_scores.points);
		let local_best_combo = this.get_storage('bestcombo');
		if (local_best_combo != null) {
			this.best_scores.combo = local_best_combo;
		}
		this.ref_el(this.elements.best_combo).html(this.best_scores.combo);
		if (this.get_storage('name') == null) {
			this.ref_el('.panel.settings').show();
		} else {
			this.config.username = this.get_storage('name');
		}
		this.ref_el('.game-username').val(this.config.username);
		if (this.get_storage('theme') == null) {
			this.set_storage('theme', 0);
		}
		this.ref_el('.game-theme').val(this.get_storage('theme'));
		this.config.theme = this.get_storage('theme');
		let local_high_scores = this.get_storage('highscores');
		if (local_high_scores != null) {
			this.highscores = JSON.parse(local_high_scores);
		}
	}

	/*
	 * Save a key with the specified value to the storage.
	 */
	set_storage (key, value) {
		localStorage[this.elements.storage + '.' + key] = value;
	}

	/*
	 * Get a key from the storage.
	 */
	get_storage (key) {
		return localStorage[this.elements.storage + '.' + key];
	}

	/*
	 * Return a DOM element referenced by the game main DOM element.
	 */
	ref_el (element) {
		if (typeof element === 'undefined') {
			return $(this.elements.application);
		} else {
			return $(this.elements.application + ' ' + element);
		}
	}

	cleanup () {
		this._level = 0;
		this._combo_best_current = 0;
		this._score_per_level = [0];
		this.scores.points = 0;
		this._cycle = 0;
		this._points = [0];
		this._evil = 0.8;
		this._colors = 3;
		for (let x = 0; x < this.config.lines; x += 1) {
			for (let y = 0; y < this.config.columns; y += 1) {
				if (this.bubble({
					x: x,
					y: y
				})) {
					this.bubble({
						x: x,
						y: y
					}).remove();
				}
			}
		}
		this.ref_el('.level').text(this._level);
	}

	init () {
		this.cleanup();
		if (this.config.username === '') {
			this.config.username = 'user-' + Math.floor(Math.random() * 9999);
			this.set_storage('name', this.config.username);
			this.ref_el('.game-username').val(this.config.username);
		}
		this.new_level();
	}

	new_level () {
		let new_bubble;
		let new_bubbles = [];
		if (this._level === 5) {
			this._colors = 4;
			this._evil = 0.7;
		} else if (this._level === 10) {
			this._colors = 5;
			this._evil = 0.6;
		} else if (this._level === 15) {
			this._colors = 6;
			this._evil = 0.5;
		}
		this._level += 1;
		$('.application .level').text(this._level);
		for (let x = 0; x < this.config.lines; x += 1) {
			for (let y = 0; y < this.config.columns; y += 1) {
				if (!this.bubble({ x: x, y: y })) {
					new_bubble = this.new_bubble({
						x: x,
						y: y,
						type: Math.random() > this._evil ? (x + y * 2) % 4 : Math.floor(Math.random() * this._colors)
					});
					new_bubble.append();
					new_bubbles.push({
						x: x,
						y: y
					});
				}
			}
		}
		this.bulkshow(new_bubbles);
		if (this.check_level_ended()) {
			setTimeout(this.game_over.apply(this), 600);
		}
		this.ref_el('.game-container').width(this.config.lines * this.config.elements.width);
	}

	set_score () {
		this.scores.points += Math.floor((this._points[this._cycle] * (this._points[this._cycle] - 1)) / 2 * (Math.log(this._level) + 1));
		this._score_per_level[this._level] = this.scores.points;
		$('.application .score').text(this.scores.points);
		$('.application .combo').text(this._points[this._cycle]);
		if (this._points[this._cycle] > this._combo_best_current) {
			this._combo_best_current = this._points[this._cycle];
		}
	}
	
	flood_check (coords) {
		let dir = {};
		let rv = 0;
		if (this.bubble({
			x: coords.x - 1,
			y: coords.y
		}) && this.bubble({
			x: coords.x - 1,
			y: coords.y
		}).type === coords.type) {
			rv += 1;
			dir = {
				x: coords.x - 1,
				y: coords.y
			};
		}
		if (this.bubble({
			x: coords.x,
			y: coords.y - 1
		}) && this.bubble({
			x: coords.x,
			y: coords.y - 1
		}).type === coords.type) {
			rv += 1;
			dir = {
				x: coords.x,
				y: coords.y - 1
			};
		}
		if (this.bubble({
			x: coords.x + 1,
			y: coords.y
		}) && this.bubble({
			x: coords.x + 1,
			y: coords.y
		}).type === coords.type) {
			rv += 1;
			dir = {
				x: coords.x + 1,
				y: coords.y
			};
		}
		if (this.bubble({
			x: coords.x,
			y: coords.y + 1
		}) && this.bubble({
			x: coords.x,
			y: coords.y + 1
		}).type === coords.type) {
			rv += 1;
			dir = {
				x: coords.x,
				y: coords.y + 1
			};
		}
		if (rv === 0) {
			return false;
		} else if (rv >= 2) {
			return true;
		} else {
			if (coords.tried) {
				return false;
			} else {
				dir.type = coords.type;
				dir.tried = true;
				return this.flood_check(dir);
			}
		}
	}
	
	bulkshow (bubbles) {
		let nphase = 5;
		let phases = [];
		for (let i = 0; i < nphase; i += 1) {
			phases[i] = [];
		}
		for (let i = 0; i < bubbles.length; i += 1) {
			phases[Math.floor(Math.random() * nphase)].push(this.bubble({
				x: bubbles[i].x,
				y: bubbles[i].y
			}));
		}
		let step = function () {
			let to_show_now;
			let over = 0;
			if (phases.length > 0) {
				do {
					to_show_now = phases.pop();
				} while (to_show_now && to_show_now.length === 0);
				if (to_show_now) {
					for (let i = 0; i < to_show_now.length; i += 1) {
						to_show_now[i].show();
					}
					setTimeout(step, 50);
				}
			}
		};
		step();
	}

	bulkfall (bubbles) {
		let self = this;
		let i = 0;
		let step = function () {
			if (i < bubbles.length + 3) {
				if (bubbles[i]) {
					for (let j = 0; j < bubbles[i].length; j += 1) {
						bubbles[i][j].style.marginTop = Math.floor(bubbles[i][j].y * self.config.elements.height - self.config.elements.height / 2) + 'px';
					}
				}
				if (bubbles[i - 1]) {
					for (let j = 0; j < bubbles[i - 1].length; j += 1) {
						bubbles[i - 1][j].style.marginTop = bubbles[i - 1][j].y * self.config.elements.height - 1 + 'px';
					}
				}
				if (bubbles[i - 2]) {
					for (let j = 0; j < bubbles[i - 2].length; j += 1) {
						bubbles[i - 2][j].style.marginTop = bubbles[i - 2][j].y * self.config.elements.height - 2 + 'px';
					}
				}
				if (bubbles[i - 3]) {
					for (let j = 0; j < bubbles[i - 3].length; j += 1) {
						bubbles[i - 3][j].style.marginTop = bubbles[i - 3][j].y * self.config.elements.height + 'px';
					}
				}
				i += 1;
				setTimeout(step, 50);
			}
		};
		step();
	}
	
	floodrem (coords) {
		let self = this;
		let color = this.bubble({
			x: coords.x,
			y: coords.y
		}).type;
		let step = function (coords_in) {
			let gb = self.bubble({
				x: coords_in.x,
				y: coords_in.y
			});
			if (gb && gb.type === color) {
				self._points[self._cycle] += 1;
				self.bubble({
					x: coords_in.x,
					y: coords_in.y
				}).remove();
				step({
					x: coords_in.x - 1,
					y: coords_in.y
				});
				step({
					x: coords_in.x,
					y: coords_in.y - 1
				});
				step({
					x: coords_in.x + 1,
					y: coords_in.y
				});
				step({
					x: coords_in.x,
					y: coords_in.y + 1
				});
			}
		};
		step(coords);
		this.fall();
		if (this.check_level_ended()) {
			setTimeout(this.new_level.apply(this), 200);
		}
	}

	fall () {
		let bubble;
		let bubbles = [];
		let bubbles_collection;
		let last_good_y;
		for (let x = 0; x < this.config.lines; x += 1) {
			bubbles_collection = [];
			last_good_y = this.config.columns - 1;
			for (let y = this.config.columns - 1; y >= 0; y -= 1) {
				bubble = this.bubble({
					x: x,
					y: y
				});
				if (bubble) {
					if (last_good_y > y) {
						bubble.fall({
							y: last_good_y
						});
						bubbles_collection.push(bubble);
					}
					last_good_y -= 1;
				}
			}
			if (bubbles_collection.length > 0) {
				bubbles.push(bubbles_collection);
			}
		}
		this.bulkfall(bubbles);
	}
	
	/*
	 * Check if the current level ended.
	 */
	check_level_ended () {
		let bubbles = document.getElementsByClassName('game-container')[0].getElementsByTagName('span');
		for (let i = 0; i < bubbles.length; i += 1) {
			if (this.flood_check({
				x: bubbles[i].x,
				y: bubbles[i].y,
				type: bubbles[i].type
			})) {
				return false;
			}
		}
		return true;
	}
	
	game_over () {
		this.best_scores.combo = this._combo_best_current;
		this.notify('GAME OVER<br />Score: ' + this.scores.points + '<br />Max combo: ' + this.scores.combo, true);
		this.save();
		this.ref_el('footer').hide();
	}

	/*
	 * Create a new bubble.
	 */
	new_bubble (spec) {
		let self = this;
		let that = document.createElement("span");
		that.className = 'element theme' + this.config.theme;
		that.style.marginLeft = spec.x * this.config.elements.width + 'px';
		that.style.marginTop = spec.y * this.config.elements.height + 'px';
		that.style.backgroundPosition = '0 -' + spec.type * this.config.elements.height + 'px';
		that.id = 'element_' + spec.x + '_' + spec.y;
		that.x = spec.x;
		that.y = spec.y;
		that.type = spec.type;
		that.append = function () {
			document.getElementsByClassName('game-container')[0].appendChild(this);
		};
		that.show = function () {
			this.style.opacity = 1;
			this.style.cursor = 'pointer';
		};
		that.remove = function (coords) {
			document.getElementsByClassName('game-container')[0].removeChild(this);
		};
		that.fall = function (where) {
			this.id = 'element_' + this.x + '_' + where.y;
			this.y = where.y;
		};
		that.onmousedown = function () {
			let filled = {};
			let color = self.bubble({
				x: this.x,
				y: this.y
			}).type;
			if (self.flood_check({
				x: this.x,
				y: this.y,
				type: this.type
			})) {
				self._points[self._cycle] = 0;
				self.floodrem({
					x: this.x,
					y: this.y
				});
				self.set_score();
				self._cycle += 1;
			}
			return false;
		};
		that.ontouchstart = that.onmousedown;
		return that;
	}

	/*
	 * Return a bubble at the specific coords
	 */
	bubble (coords) {
		if (coords.x >= 0 && coords.y >= 0 && coords.x < this.config.lines && coords.y < this.config.columns) {
			if (document.getElementById('element_' + coords.x + '_' + coords.y)) {
				return document.getElementById('element_' + coords.x + '_' + coords.y);
			}
		}
	}

	/*
	 * Reset the game UI to initial state.
	 */
	reset_ui () {
		this.ref_el('.content').hide();
		this.ref_el(this.elements.notification).html('');
		this.ref_el(this.elements.score).html('0');
		this.ref_el(this.elements.combo).html('0');
		this.ref_el('footer').hide();
	};

	/*
	 * Restart the game.
	 */
	restart () {
		this.ref_el('footer').show();
		$('.application .content').hide();
		this.init();
	}

	/*
	 * Stop the game.
	 */
	stop () {
		this.ref_el(this.elements.stop).removeClass('attention').html('New Game');
		this.reset_ui();
		this.ref_el('.panel').hide();
		this.ref_el('.content').fadeIn();
		this._restart_mode = false;
		clearTimeout(this.restart_timeout);
	}

	/*
	 * Perform a game notification.
	 */
	notify (message) {
		this.ref_el(this.elements.notification).html('<div>' + message + '</div>');
	}
}

$(document).ready(function() {
	new game();
});
